jQuery(document).ready(function($) {
    const bundleOptions = $('.bundle-option');
    const totalPriceElement = $('.total-price');
    const addToCartButton = $('.add-to-cart-bundle');

    bundleOptions.on('click', function() {
        bundleOptions.removeClass('selected');
        $(this).addClass('selected');

        // Update main product image with bundle image
        var imageUrl = $(this).data('image');

        if (imageUrl) {
            updateProductImage(imageUrl);
        }

        const quantity = $(this).data('quantity');
        const discount = $(this).data('discount');
        const basePrice = $(this).data('price');

        console.log("Raw discount value:", discount, "Type:", typeof discount);

        const originalPrice = basePrice * quantity;

        // Convert discount to float and check if it's meaningful (>= 0.01%)
        // This ensures 0, "0", 0.00, empty string, null, undefined, false, NaN don't show the discount badge
        let discountFloat = 0;
        if (discount !== '' && discount !== null && discount !== undefined) {
            const parsed = parseFloat(discount);
            if (!isNaN(parsed)) {
                discountFloat = parsed;
            }
        }
        console.log("Parsed discount:", discountFloat);

        const hasDiscount = discountFloat >= 0.01;
        console.log("Has discount:", hasDiscount);

        const discountedPrice = originalPrice * (1 - discountFloat / 100);

        // Build HTML conditionally based on discount availability
        let priceHTML = '';

        // Show original price only if there's a discount
        if (hasDiscount) {
            priceHTML += `<span class="original-price">${formatPrice(originalPrice)}</span>`;
        }

        // Always show the final price
        priceHTML += `<span class="discounted-price">${formatPrice(discountedPrice)}</span>`;

        // Show saving badge only if there's a meaningful discount (>= 0.01%)
        if (hasDiscount) {
            priceHTML += `<span class="saving">Saving ${discountFloat.toFixed(2)}%</span>`;
        }

        totalPriceElement.html(priceHTML);
    });

    /**
     * Update the main product image to display the selected bundle image
     * This function works with WooCommerce's product gallery and handles multiple theme variations
     *
     * @param {string} imageUrl - The URL of the bundle image to display
     */
    function updateProductImage(imageUrl) {
        if (!imageUrl) {
            return;
        }

        // Normalize the image URL (remove size suffixes for comparison)
        var normalizedUrl = imageUrl.replace(/-\d+x\d+(?=\.[a-z]{3,4}$)/i, '');

        // Method 1: Try to find and click the matching thumbnail in the gallery
        var foundThumbnail = false;
        var $gallery = $('.woocommerce-product-gallery');

        if ($gallery.length) {
            // Look for thumbnails in flex-control-thumbs (Flexslider - common WooCommerce gallery)
            var $thumbnails = $gallery.find('.flex-control-thumbs li, .flex-control-nav li');

            $thumbnails.each(function() {
                var $thumb = $(this).find('img');
                if (!$thumb.length) {
                    return true; // Continue to next iteration
                }

                var thumbSrc = $thumb.attr('src') || '';
                var thumbDataSrc = $thumb.attr('data-src') || '';

                // Normalize thumbnail URLs for comparison
                var normalizedThumbSrc = thumbSrc.replace(/-\d+x\d+(?=\.[a-z]{3,4}$)/i, '');
                var normalizedThumbDataSrc = thumbDataSrc.replace(/-\d+x\d+(?=\.[a-z]{3,4}$)/i, '');

                // Check if this thumbnail matches our bundle image
                if (normalizedThumbSrc === normalizedUrl ||
                    normalizedThumbDataSrc === normalizedUrl ||
                    thumbSrc === imageUrl ||
                    thumbDataSrc === imageUrl) {

                    // Click the thumbnail to trigger WooCommerce's built-in image switching
                    $thumb.trigger('click');
                    foundThumbnail = true;
                    console.log('Bundle image found in gallery thumbnails and clicked');
                    return false; // Break the loop
                }
            });
        }

        // Method 2: If thumbnail click didn't work, try finding the image in gallery slides
        if (!foundThumbnail && $gallery.length) {
            var $slides = $gallery.find('.woocommerce-product-gallery__image, .flex-viewport .slides > li');

            $slides.each(function(index) {
                var $slideImg = $(this).find('img');
                if (!$slideImg.length) {
                    return true; // Continue
                }

                var slideSrc = $slideImg.attr('src') || '';
                var slideDataSrc = $slideImg.attr('data-large_image') || $slideImg.attr('data-src') || '';

                var normalizedSlideSrc = slideSrc.replace(/-\d+x\d+(?=\.[a-z]{3,4}$)/i, '');
                var normalizedSlideDataSrc = slideDataSrc.replace(/-\d+x\d+(?=\.[a-z]{3,4}$)/i, '');

                if (normalizedSlideSrc === normalizedUrl ||
                    normalizedSlideDataSrc === normalizedUrl ||
                    slideSrc === imageUrl ||
                    slideDataSrc === imageUrl) {

                    // Trigger the corresponding thumbnail if Flexslider is present
                    if (typeof $.fn.flexslider !== 'undefined' && $gallery.data('flexslider')) {
                        $gallery.flexslider(index);
                        console.log('Bundle image displayed via Flexslider');
                    } else {
                        // Fallback: directly update the main image
                        updateMainImageDirectly(imageUrl);
                    }
                    foundThumbnail = true;
                    return false; // Break
                }
            });
        }

        // Method 3: Fallback - directly update the main product image if gallery methods failed
        if (!foundThumbnail) {
            updateMainImageDirectly(imageUrl);
            console.log('Bundle image updated directly (gallery method not available)');
        }
    }

    /**
     * Directly update the main product image source
     * Used as a fallback when WooCommerce gallery methods are not available
     *
     * @param {string} imageUrl - The URL of the image to display
     */
    function updateMainImageDirectly(imageUrl) {
        // Find the main product image
        var $mainImage = $('.woocommerce-product-gallery__image:first img, .woocommerce-main-image img, .product-image img, .wp-post-image').first();

        if ($mainImage.length) {
            // Update the image source with a smooth transition
            $mainImage.fadeOut(200, function() {
                $(this).attr('src', imageUrl);

                // Also update srcset and data attributes if they exist
                if ($(this).attr('srcset')) {
                    $(this).attr('srcset', imageUrl);
                }
                if ($(this).attr('data-large_image')) {
                    $(this).attr('data-large_image', imageUrl);
                }
                if ($(this).attr('data-src')) {
                    $(this).attr('data-src', imageUrl);
                }

                $(this).fadeIn(200);
            });

            // Also update the parent link if it's a zoom/lightbox link
            var $parentLink = $mainImage.closest('a');
            if ($parentLink.length && $parentLink.attr('href')) {
                $parentLink.attr('href', imageUrl);
            }
        }
    }

    addToCartButton.on('click', function(e) {
        e.preventDefault();
        const selectedBundle = $('.bundle-option.selected');
        if (selectedBundle.length === 0) {
            alert('Please select a package');
            return;
        }

        // const productId = $('input[name="product_id"]').val();

        const productId = $(this).data('productid');

        console.log('productId '+productId);

        const index = selectedBundle.data('index');
        // const discount = selectedBundle.data('discount');



        $.ajax({
            url: wc_add_to_cart_params.ajax_url,
            type: 'POST',
            data: {
                action: 'add_bundle_to_cart',
                product_id: productId,
                bundle_index: index
            },
            success: function(response) {
                if (response.success) {
                    window.location.href = wc_add_to_cart_params.cart_url;
                } else {
                    alert('Error adding to cart. Please try again.');
                }
            }
        });
    });

    function formatPrice(price) {
        // Use WooCommerce currency settings from localized script
        if (typeof kbq_currency_params === 'undefined') {
            return '$' + price.toFixed(2); // Fallback
        }

        var decimals = parseInt(kbq_currency_params.number_of_decimals);
        var decimalSep = kbq_currency_params.decimal_separator;
        var thousandSep = kbq_currency_params.thousand_separator;
        var symbol = kbq_currency_params.currency_symbol;
        var position = kbq_currency_params.currency_position;

        // Format the number with decimals
        var formattedPrice = price.toFixed(decimals);

        // Split into whole and decimal parts
        var parts = formattedPrice.split('.');
        var wholePart = parts[0];
        var decimalPart = parts[1] || '';

        // Add thousand separators
        wholePart = wholePart.replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);

        // Combine with decimal separator
        formattedPrice = decimalPart ? wholePart + decimalSep + decimalPart : wholePart;

        // Add currency symbol based on position
        switch(position) {
            case 'left':
                return symbol + formattedPrice;
            case 'right':
                return formattedPrice + symbol;
            case 'left_space':
                return symbol + ' ' + formattedPrice;
            case 'right_space':
                return formattedPrice + ' ' + symbol;
            default:
                return symbol + formattedPrice;
        }
    }

    // Select the first option by default
    bundleOptions.first().click();
});

/*

jQuery(document).ready(function($) {
    // Handle bundle selection
    $('#bundle-quantity-options .bundle-option').on('click', function() {
        $('.bundle-option').removeClass('selected');
        $(this).addClass('selected');

        var quantity = $(this).data('quantity');
        var discount = $(this).data('discount');
        var originalPrice = parseFloat($('#price').data('original-price'));

        // Validate and convert discount to float (handle empty strings, null, undefined)
        var discountFloat = 0;
        if (discount !== '' && discount !== null && discount !== undefined) {
            var parsed = parseFloat(discount);
            if (!isNaN(parsed)) {
                discountFloat = parsed;
            }
        }

        // Calculate the new price
        var discountedPrice = originalPrice - (originalPrice * (discountFloat / 100));
        var totalPrice = discountedPrice * quantity;

        // Update the price display
        $('#bundle-price-display').text('Total: $' + totalPrice.toFixed(2));

        // Store selected bundle details in hidden inputs
        $('#bundle-quantity').val(quantity);
        $('#bundle-discount').val(discountFloat);
    });

    // Add the bundle to the cart
    $('form.cart').on('submit', function(e) {
        e.preventDefault();

        var quantity = $('#bundle-quantity').val();
        var discount = $('#bundle-discount').val();

        if (!quantity) {
            alert('Please select a bundle option');
            return;
        }

        var form = $(this);

        // Adjust quantity before submitting the form
        form.find('input.qty').val(quantity);

        // Submit the form
        form.unbind('submit').submit();
    });
});
*/
