jQuery(document).ready(function($) {
    console.log('KBQ Settings JS loaded');
    console.log('kbqSettings:', kbqSettings);

    // Check if reset button exists
    if ($('#kbq-reset-colors').length) {
        console.log('Reset colors button found in DOM');
    } else {
        console.warn('Reset colors button NOT found in DOM');
    }

    // Initialize color pickers
    $('.kbq-color-picker').wpColorPicker({
        change: function(event, ui) {
            const $input = $(this);
            const color = ui.color.toString();

            // Update the input value
            $input.val(color);

            // Check contrast for specific fields
            checkContrastRatio($input);
        },
        clear: function() {
            const $input = $(this);
            const defaultColor = $input.data('default-color');
            $input.val(defaultColor);
        }
    });

    // Auto-detect theme colors button
    $('#kbq-detect-theme-colors').on('click', function(e) {
        e.preventDefault();
        const $button = $(this);
        const originalText = $button.text();

        $button.prop('disabled', true).text(kbqSettings.strings.detecting);

        $.ajax({
            url: kbqSettings.ajax_url,
            type: 'POST',
            data: {
                action: 'kbq_detect_theme_colors',
                nonce: kbqSettings.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Update color picker values
                    $.each(response.data.colors, function(key, value) {
                        const $input = $('#kbq_color_' + key);
                        if ($input.length) {
                            $input.wpColorPicker('color', value);
                        }
                    });

                    // Show success message
                    $('<div class="notice notice-success is-dismissible"><p>' + kbqSettings.strings.detected + '</p></div>')
                        .insertAfter('.wrap h1')
                        .delay(3000)
                        .fadeOut();
                } else {
                    // Show error message
                    $('<div class="notice notice-warning is-dismissible"><p>' + kbqSettings.strings.detection_failed + '</p></div>')
                        .insertAfter('.wrap h1')
                        .delay(5000)
                        .fadeOut();
                }
            },
            error: function() {
                $('<div class="notice notice-error is-dismissible"><p>' + kbqSettings.strings.detection_failed + '</p></div>')
                    .insertAfter('.wrap h1')
                    .delay(5000)
                    .fadeOut();
            },
            complete: function() {
                $button.prop('disabled', false).text(originalText);
            }
        });
    });

    // Reset colors to default button
    $('#kbq-reset-colors').on('click', function(e) {
        // Only use AJAX if kbqSettings is available (JavaScript working properly)
        if (typeof kbqSettings === 'undefined' || !kbqSettings.reset_nonce) {
            console.log('kbqSettings not available, using fallback link behavior');
            return true; // Allow the link to work normally
        }

        e.preventDefault(); // Prevent link from navigating
        const $button = $(this);
        const originalText = $button.text();

        console.log('Reset colors button clicked');

        // Confirmation is handled by onclick attribute, but we can skip it here since it already ran
        console.log('Starting color reset...');
        $button.css('pointer-events', 'none').css('opacity', '0.5').text(kbqSettings.strings.resetting);

        $.ajax({
            url: kbqSettings.ajax_url,
            type: 'POST',
            data: {
                action: 'kbq_reset_colors_to_default',
                nonce: kbqSettings.reset_nonce
            },
            success: function(response) {
                console.log('AJAX Response:', response);

                if (response.success) {
                    console.log('Colors reset successfully!');
                    console.log('New colors:', response.data.colors);

                    // Update color picker values with defaults
                    $.each(response.data.colors, function(key, value) {
                        const $input = $('#kbq_color_' + key);
                        if ($input.length) {
                            console.log('Updating color for ' + key + ' to ' + value);
                            $input.wpColorPicker('color', value);
                        }
                    });

                    // Remove any existing notices first
                    $('.kbq-notification-message').remove();

                    // Show success notification with animation
                    const $notice = $('<div class="notice notice-success is-dismissible kbq-notification-message" style="display:none;"><p><strong>' + kbqSettings.strings.reset_success + '</strong></p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button></div>');

                    // Insert and show with animation
                    $notice.insertAfter('.wrap h1').slideDown(300);
                    console.log('Notification displayed');

                    // Auto-dismiss after 4 seconds with fade out
                    setTimeout(function() {
                        $notice.fadeOut(400, function() {
                            $(this).remove();
                        });
                    }, 4000);

                    // Add dismissible button functionality
                    $notice.on('click', '.notice-dismiss', function(e) {
                        e.preventDefault();
                        $notice.fadeOut(300, function() {
                            $(this).remove();
                        });
                    });

                    // Scroll to the notification
                    $('html, body').animate({
                        scrollTop: $notice.offset().top - 100
                    }, 300);
                } else {
                    console.error('Reset failed:', response);
                    // Show error message
                    const errorMessage = response.data && response.data.message
                        ? response.data.message
                        : kbqSettings.strings.reset_failed;

                    // Remove any existing notices first
                    $('.kbq-notification-message').remove();

                    const $errorNotice = $('<div class="notice notice-error is-dismissible kbq-notification-message"><p>' + errorMessage + '</p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button></div>');
                    $errorNotice.insertAfter('.wrap h1');

                    setTimeout(function() {
                        $errorNotice.fadeOut(400, function() {
                            $(this).remove();
                        });
                    }, 5000);

                    $errorNotice.on('click', '.notice-dismiss', function(e) {
                        e.preventDefault();
                        $errorNotice.fadeOut(300, function() {
                            $(this).remove();
                        });
                    });
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error:', status, error);
                console.error('Response:', xhr.responseText);

                $('.kbq-notification-message').remove();

                const $errorNotice = $('<div class="notice notice-error is-dismissible kbq-notification-message"><p>' + kbqSettings.strings.reset_failed + '</p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button></div>');
                $errorNotice.insertAfter('.wrap h1');

                setTimeout(function() {
                    $errorNotice.fadeOut(400, function() {
                        $(this).remove();
                    });
                }, 5000);

                $errorNotice.on('click', '.notice-dismiss', function(e) {
                    e.preventDefault();
                    $errorNotice.fadeOut(300, function() {
                        $(this).remove();
                    });
                });
            },
            complete: function() {
                console.log('AJAX request completed');
                $button.css('pointer-events', '').css('opacity', '').text(originalText);
            }
        });
    });

    // Contrast ratio checker
    function checkContrastRatio($input) {
        const inputId = $input.attr('id');

        // Only check specific combinations
        if (inputId === 'kbq_color_button_text') {
            const textColor = $input.val();
            const bgColor = $('#kbq_color_button_bg').val();

            if (textColor && bgColor) {
                const ratio = calculateContrastRatio(textColor, bgColor);
                const $warning = $('#kbq_button_contrast_warning');

                if (ratio < 4.5) {
                    $warning.find('.message').text(
                        kbqSettings.strings.contrast_warning.replace('%s', ratio.toFixed(2))
                    );
                    $warning.show();
                } else {
                    $warning.hide();
                }
            }
        }
    }

    // Helper function to calculate contrast ratio
    function calculateContrastRatio(color1, color2) {
        const l1 = getRelativeLuminance(color1);
        const l2 = getRelativeLuminance(color2);

        const lighter = Math.max(l1, l2);
        const darker = Math.min(l1, l2);

        return (lighter + 0.05) / (darker + 0.05);
    }

    function getRelativeLuminance(hex) {
        hex = hex.replace('#', '');
        const r = parseInt(hex.substr(0, 2), 16) / 255;
        const g = parseInt(hex.substr(2, 2), 16) / 255;
        const b = parseInt(hex.substr(4, 2), 16) / 255;

        const rsRGB = r <= 0.03928 ? r / 12.92 : Math.pow((r + 0.055) / 1.055, 2.4);
        const gsRGB = g <= 0.03928 ? g / 12.92 : Math.pow((g + 0.055) / 1.055, 2.4);
        const bsRGB = b <= 0.03928 ? b / 12.92 : Math.pow((b + 0.055) / 1.055, 2.4);

        return 0.2126 * rsRGB + 0.7152 * gsRGB + 0.0722 * bsRGB;
    }

    // Check contrast on button background color change too
    $('#kbq_color_button_bg').on('change', function() {
        checkContrastRatio($('#kbq_color_button_text'));
    });

    // Initial contrast check
    checkContrastRatio($('#kbq_color_button_text'));
});
