<?php
/**
 * License Manager Class
 *
 * Handles premium license validation and feature access control
 *
 * @package Product_Bundle_Quantity
 * @since 2.5.0
 */

if (!defined('ABSPATH')) exit;

class KBQ_License_Manager {

    /**
     * License option name
     */
    const LICENSE_OPTION = 'kbq_license_data';

    /**
     * API endpoint for license validation
     */
    const API_ENDPOINT = 'https://codemoksha.com/api/license/validate';

    /**
     * Transient cache duration (12 hours)
     */
    const CACHE_DURATION = 43200;

    /**
     * Premium features list
     */
    const PREMIUM_FEATURES = array(
        'custom_colors' => 'Custom Colors',
        'product_discounts' => 'Product Discounts',
    );

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Cached license status
     */
    private $license_status = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_license_menu'), 99);
        add_action('admin_init', array($this, 'register_license_settings'));
        add_action('admin_notices', array($this, 'display_license_notices'));

        // AJAX handlers
        add_action('wp_ajax_kbq_activate_license', array($this, 'ajax_activate_license'));
        add_action('wp_ajax_kbq_deactivate_license', array($this, 'ajax_deactivate_license'));
        add_action('wp_ajax_kbq_check_license_status', array($this, 'ajax_check_license_status'));
        add_action('wp_ajax_kbq_clear_license_cache', array($this, 'ajax_clear_license_cache'));

        // Daily license check cron
        add_action('kbq_daily_license_check', array($this, 'check_license_validity'));
        if (!wp_next_scheduled('kbq_daily_license_check')) {
            wp_schedule_event(time(), 'daily', 'kbq_daily_license_check');
        }
    }

    /**
     * Add license settings page to admin menu
     */
    public function add_license_menu() {
        add_submenu_page(
            'product-bundle-quantity',                                    // Parent slug (our plugin menu)
            __('Bundle Quantities License', 'product-bundle-quantity'),  // Page title
            __('License', 'product-bundle-quantity'),                    // Menu title
            'manage_woocommerce',                                        // Capability
            'kbq-license',                                               // Menu slug
            array($this, 'render_license_page')                          // Callback
        );
    }

    /**
     * Register license settings
     */
    public function register_license_settings() {
        register_setting(
            'kbq_license_settings',
            self::LICENSE_OPTION,
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_license_data'),
                'default' => array()
            )
        );
    }

    /**
     * Sanitize license data
     */
    public function sanitize_license_data($input) {
        // If input is not an array, return empty array
        if (!is_array($input)) {
            return array();
        }

        $sanitized = array();

        // Sanitize license key
        if (isset($input['license_key'])) {
            $sanitized['license_key'] = sanitize_text_field($input['license_key']);
        }

        // Sanitize and preserve all license fields from input
        if (isset($input['status'])) {
            $sanitized['status'] = sanitize_text_field($input['status']);
        }

        if (isset($input['activated_at'])) {
            $sanitized['activated_at'] = sanitize_text_field($input['activated_at']);
        }

        if (isset($input['last_checked'])) {
            $sanitized['last_checked'] = sanitize_text_field($input['last_checked']);
        }

        if (isset($input['purchase_date'])) {
            $sanitized['purchase_date'] = sanitize_text_field($input['purchase_date']);
        }

        if (isset($input['expires_at'])) {
            $sanitized['expires_at'] = sanitize_text_field($input['expires_at']);
        }

        // Preserve features array (sanitize each element)
        if (isset($input['features']) && is_array($input['features'])) {
            $sanitized['features'] = array_map('sanitize_text_field', $input['features']);
        }

        // Preserve email and product if they exist
        if (isset($input['email'])) {
            $sanitized['email'] = sanitize_email($input['email']);
        }

        if (isset($input['product'])) {
            $sanitized['product'] = sanitize_text_field($input['product']);
        }

        return $sanitized;
    }

    /**
     * Render license page
     */
    public function render_license_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'product-bundle-quantity'));
        }

        $license_data = $this->get_license_data();
        $is_active = $this->is_license_active();

        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="kbq-license-container" style="max-width: 800px;">

                <!-- License Status Card -->
                <div class="kbq-license-status-card" style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin: 20px 0;">
                    <h2 style="margin-top: 0;"><?php _e('License Status', 'product-bundle-quantity'); ?></h2>

                    <?php if ($is_active): ?>
                        <div class="notice notice-success inline" style="margin: 10px 0; padding: 10px;">
                            <p style="margin: 0;">
                                <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                                <strong><?php _e('License Active', 'product-bundle-quantity'); ?></strong>
                            </p>
                        </div>

                        <table class="form-table" style="margin-top: 20px;">
                            <tr>
                                <th scope="row"><?php _e('License Key', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <code><?php echo esc_html($this->mask_license_key($license_data['license_key'])); ?></code>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Status', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <span style="color: #46b450; font-weight: 600; text-transform: uppercase;">
                                        <?php echo esc_html($license_data['status']); ?>
                                    </span>
                                </td>
                            </tr>
                            <?php if (!empty($license_data['purchase_date'])): ?>
                            <tr>
                                <th scope="row"><?php _e('Purchase Date', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <?php
                                    $purchase_date = strtotime($license_data['purchase_date']);
                                    echo esc_html(date_i18n(get_option('date_format'), $purchase_date));
                                    ?>
                                </td>
                            </tr>
                            <?php endif; ?>
                            <?php if (!empty($license_data['expires_at'])): ?>
                            <tr>
                                <th scope="row"><?php _e('Expiration Date', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <?php
                                    $expires = strtotime($license_data['expires_at']);
                                    echo esc_html(date_i18n(get_option('date_format'), $expires));

                                    $days_remaining = ceil(($expires - time()) / DAY_IN_SECONDS);
                                    if ($days_remaining > 0 && $days_remaining < 30) {
                                        echo '<span style="color: #d63638; margin-left: 10px; font-weight: 600;">(' . sprintf(__('%d days remaining', 'product-bundle-quantity'), $days_remaining) . ')</span>';
                                    } elseif ($days_remaining <= 0) {
                                        echo '<span style="color: #d63638; margin-left: 10px; font-weight: 600;">(' . __('Expired', 'product-bundle-quantity') . ')</span>';
                                    }
                                    ?>
                                </td>
                            </tr>
                            <?php endif; ?>
                            <?php if (!empty($license_data['features']) && is_array($license_data['features'])): ?>
                            <tr>
                                <th scope="row"><?php _e('Active Features', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <?php
                                    $features = $license_data['features'];
                                    echo '<ul style="margin: 0; padding-left: 20px;">';
                                    foreach ($features as $feature) {
                                        echo '<li>' . esc_html(ucwords(str_replace('_', ' ', $feature))) . '</li>';
                                    }
                                    echo '</ul>';
                                    ?>
                                </td>
                            </tr>
                            <?php endif; ?>
                            <tr>
                                <th scope="row"><?php _e('Last Verified', 'product-bundle-quantity'); ?></th>
                                <td>
                                    <?php
                                    if (!empty($license_data['last_checked'])) {
                                        echo esc_html(human_time_diff(strtotime($license_data['last_checked']))) . ' ' . __('ago', 'product-bundle-quantity');
                                    } else {
                                        _e('Never', 'product-bundle-quantity');
                                    }
                                    ?>
                                </td>
                            </tr>
                        </table>

                        <div style="margin-top: 20px;">
                            <button type="button" id="kbq-deactivate-license" class="button button-secondary">
                                <?php _e('Deactivate License', 'product-bundle-quantity'); ?>
                            </button>
                            <button type="button" id="kbq-check-license" class="button button-secondary" style="margin-left: 10px;">
                                <?php _e('Check License Status', 'product-bundle-quantity'); ?>
                            </button>
                            <button type="button" id="kbq-refresh-license" class="button button-secondary" style="margin-left: 10px;">
                                <?php _e('Refresh License Cache', 'product-bundle-quantity'); ?>
                            </button>
                        </div>

                    <?php else: ?>
                        <div class="notice notice-warning inline" style="margin: 10px 0; padding: 10px;">
                            <p style="margin: 0;">
                                <span class="dashicons dashicons-warning" style="color: #dba617;"></span>
                                <strong><?php _e('No Active License', 'product-bundle-quantity'); ?></strong>
                            </p>
                        </div>

                        <p><?php _e('Activate your premium license to unlock advanced features:', 'product-bundle-quantity'); ?></p>
                        <ul style="list-style: disc; margin-left: 20px;">
                            <li><?php _e('Custom color schemes for frontend elements', 'product-bundle-quantity'); ?></li>
                            <li><?php _e('Product bundle discounts', 'product-bundle-quantity'); ?></li>
                            <li><?php _e('Priority support', 'product-bundle-quantity'); ?></li>
                        </ul>

                        <form id="kbq-license-form" style="margin-top: 20px;">
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="kbq_license_key"><?php _e('License Key', 'product-bundle-quantity'); ?></label>
                                    </th>
                                    <td>
                                        <input type="text"
                                               id="kbq_license_key"
                                               name="license_key"
                                               class="regular-text"
                                               placeholder="XXXX-XXXX-XXXX-XXXX"
                                               value="<?php echo esc_attr(!empty($license_data['license_key']) ? $license_data['license_key'] : ''); ?>" />
                                        <p class="description">
                                            <?php printf(
                                                __('Enter your license key from %s', 'product-bundle-quantity'),
                                                '<a href="https://www.codemoksha.com/product-bundle-quantities" target="_blank">codemoksha.com</a>'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>
                            </table>

                            <button type="submit" id="kbq-activate-license" class="button button-primary">
                                <?php _e('Activate License', 'product-bundle-quantity'); ?>
                            </button>
                        </form>
                    <?php endif; ?>
                </div>

                <!-- Premium Features -->
                <div class="kbq-premium-features" style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin: 20px 0;">
                    <h2 style="margin-top: 0;"><?php _e('Premium Features', 'product-bundle-quantity'); ?></h2>

                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php _e('Feature', 'product-bundle-quantity'); ?></th>
                                <th><?php _e('Status', 'product-bundle-quantity'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (self::PREMIUM_FEATURES as $feature_key => $feature_name): ?>
                            <tr>
                                <td><strong><?php echo esc_html($feature_name); ?></strong></td>
                                <td>
                                    <?php if ($this->has_premium_access($feature_key)): ?>
                                        <span style="color: #46b450;">
                                            <span class="dashicons dashicons-yes"></span> <?php _e('Enabled', 'product-bundle-quantity'); ?>
                                        </span>
                                    <?php else: ?>
                                        <span style="color: #d63638;">
                                            <span class="dashicons dashicons-lock"></span> <?php _e('Requires Premium License', 'product-bundle-quantity'); ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Support Section -->
                <div class="kbq-support-section" style="background: #f0f6fc; border: 1px solid #c3d9f0; border-radius: 4px; padding: 20px; margin: 20px 0;">
                    <h3 style="margin-top: 0;"><?php _e('Need Help?', 'product-bundle-quantity'); ?></h3>
                    <p><?php _e('Visit our support portal or contact us for assistance.', 'product-bundle-quantity'); ?></p>
                    <a href="https://codemoksha.com/support" target="_blank" class="button button-secondary">
                        <?php _e('Get Support', 'product-bundle-quantity'); ?>
                    </a>
                </div>
            </div>
        </div>

        <style>
            .kbq-license-container .notice.inline {
                display: block;
            }
            #kbq-license-form .spinner {
                float: none;
                margin: 0 10px;
            }
        </style>

        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Activate license
            $('#kbq-license-form').on('submit', function(e) {
                e.preventDefault();

                var $button = $('#kbq-activate-license');
                var $form = $(this);
                var licenseKey = $('#kbq_license_key').val().trim();

                if (!licenseKey) {
                    alert('<?php _e('Please enter a license key.', 'product-bundle-quantity'); ?>');
                    return;
                }

                $button.prop('disabled', true).text('<?php _e('Activating...', 'product-bundle-quantity'); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kbq_activate_license',
                        license_key: licenseKey,
                        nonce: '<?php echo wp_create_nonce('kbq_license_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php _e('License activation failed.', 'product-bundle-quantity'); ?>');
                            $button.prop('disabled', false).text('<?php _e('Activate License', 'product-bundle-quantity'); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php _e('An error occurred. Please try again.', 'product-bundle-quantity'); ?>');
                        $button.prop('disabled', false).text('<?php _e('Activate License', 'product-bundle-quantity'); ?>');
                    }
                });
            });

            // Deactivate license
            $('#kbq-deactivate-license').on('click', function(e) {
                e.preventDefault();

                if (!confirm('<?php _e('Are you sure you want to deactivate this license?', 'product-bundle-quantity'); ?>')) {
                    return;
                }

                var $button = $(this);
                $button.prop('disabled', true).text('<?php _e('Deactivating...', 'product-bundle-quantity'); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kbq_deactivate_license',
                        nonce: '<?php echo wp_create_nonce('kbq_license_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php _e('License deactivation failed.', 'product-bundle-quantity'); ?>');
                            $button.prop('disabled', false).text('<?php _e('Deactivate License', 'product-bundle-quantity'); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php _e('An error occurred. Please try again.', 'product-bundle-quantity'); ?>');
                        $button.prop('disabled', false).text('<?php _e('Deactivate License', 'product-bundle-quantity'); ?>');
                    }
                });
            });

            // Check license status
            $('#kbq-check-license').on('click', function(e) {
                e.preventDefault();

                var $button = $(this);
                $button.prop('disabled', true).text('<?php _e('Checking...', 'product-bundle-quantity'); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kbq_check_license_status',
                        nonce: '<?php echo wp_create_nonce('kbq_license_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php _e('License check failed.', 'product-bundle-quantity'); ?>');
                        }
                        $button.prop('disabled', false).text('<?php _e('Check License Status', 'product-bundle-quantity'); ?>');
                    },
                    error: function() {
                        alert('<?php _e('An error occurred. Please try again.', 'product-bundle-quantity'); ?>');
                        $button.prop('disabled', false).text('<?php _e('Check License Status', 'product-bundle-quantity'); ?>');
                    }
                });
            });

            // Refresh license cache
            $('#kbq-refresh-license').on('click', function(e) {
                e.preventDefault();

                var $button = $(this);
                $button.prop('disabled', true).text('<?php _e('Refreshing...', 'product-bundle-quantity'); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kbq_clear_license_cache',
                        nonce: '<?php echo wp_create_nonce('kbq_license_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php _e('License refresh failed.', 'product-bundle-quantity'); ?>');
                        }
                        $button.prop('disabled', false).text('<?php _e('Refresh License Cache', 'product-bundle-quantity'); ?>');
                    },
                    error: function() {
                        alert('<?php _e('An error occurred. Please try again.', 'product-bundle-quantity'); ?>');
                        $button.prop('disabled', false).text('<?php _e('Refresh License Cache', 'product-bundle-quantity'); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX: Activate license
     */
    public function ajax_activate_license() {
        check_ajax_referer('kbq_license_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'product-bundle-quantity')));
        }

        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';

        if (empty($license_key)) {
            wp_send_json_error(array('message' => __('Please provide a license key.', 'product-bundle-quantity')));
        }

        $result = $this->activate_license($license_key);

        if ($result['success']) {
            wp_send_json_success(array('message' => $result['message']));
        } else {
            wp_send_json_error(array('message' => $result['message']));
        }
    }

    /**
     * AJAX: Deactivate license
     */
    public function ajax_deactivate_license() {
        check_ajax_referer('kbq_license_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'product-bundle-quantity')));
        }

        $result = $this->deactivate_license();

        if ($result['success']) {
            wp_send_json_success(array('message' => $result['message']));
        } else {
            wp_send_json_error(array('message' => $result['message']));
        }
    }

    /**
     * AJAX: Check license status
     */
    public function ajax_check_license_status() {
        check_ajax_referer('kbq_license_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'product-bundle-quantity')));
        }

        $this->check_license_validity();

        $license_data = $this->get_license_data();

        if ($license_data['status'] === 'active') {
            wp_send_json_success(array('message' => __('License is active and valid.', 'product-bundle-quantity')));
        } else {
            wp_send_json_error(array('message' => __('License is not active or has expired.', 'product-bundle-quantity')));
        }
    }

    /**
     * AJAX: Clear license cache and force revalidation
     */
    public function ajax_clear_license_cache() {
        check_ajax_referer('kbq_license_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'product-bundle-quantity')));
        }

        $license_data = $this->get_license_data();

        if (empty($license_data['license_key'])) {
            wp_send_json_error(array('message' => __('No license key found to refresh.', 'product-bundle-quantity')));
        }

        // Clear all license caches
        delete_transient('kbq_license_cache_' . md5($license_data['license_key'] . get_site_url()));
        $this->license_status = null;

        // Force a fresh check
        $this->check_license_validity();

        wp_send_json_success(array(
            'message' => __('License cache cleared and revalidated successfully.', 'product-bundle-quantity')
        ));
    }

    /**
     * Activate license
     *
     * @param string $license_key The license key to activate
     * @return array Result array with 'success' and 'message'
     */
    public function activate_license($license_key) {
        // Validate license with API
        $validation_result = $this->validate_license_with_api($license_key, 'activate');

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('KBQ License Activation: API Result = ' . print_r($validation_result, true));
        }

        if (!$validation_result['success']) {
            // Log the failure
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('KBQ License Activation FAILED: ' . $validation_result['message']);
            }

            return array(
                'success' => false,
                'message' => $validation_result['message']
            );
        }

        // Extract license data from API response
        $api_data = $validation_result['data'] ?? array();

        // CRITICAL FIX: Ensure status is ALWAYS set to 'active' on successful validation
        // The API validation succeeded, so we grant access regardless of what the API returns
        $license_data = array(
            'license_key'     => $license_key,
            'status'          => 'active',  // Force active on successful API validation
            'activated_at'    => current_time('mysql'),
            'last_checked'    => current_time('mysql'),
            'purchase_date'   => $api_data['purchaseDate'] ?? '',
            'expires_at'      => $api_data['expirationDate'] ?? '',
            'features'        => $api_data['features'] ?? array(),
        );

        update_option(self::LICENSE_OPTION, $license_data);

        // CRITICAL DEBUG: Immediately verify what was saved
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            $immediately_read_back = get_option(self::LICENSE_OPTION, array());
            error_log('KBQ BEFORE SAVE: Attempting to save = ' . print_r($license_data, true));
            error_log('KBQ AFTER SAVE: Immediately read back = ' . print_r($immediately_read_back, true));
            error_log('KBQ SAVE VERIFICATION: Match = ' . ($license_data === $immediately_read_back ? 'YES' : 'NO'));
        }

        // Clear cache
        $this->license_status = null;
        delete_transient('kbq_license_cache_' . md5($license_key . get_site_url()));

        // Debug log success
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('KBQ License Activation SUCCESS: Saved data = ' . print_r($license_data, true));
        }

        return array(
            'success' => true,
            'message' => $validation_result['message'] ?? __('License activated successfully!', 'product-bundle-quantity')
        );
    }

    /**
     * Deactivate license
     *
     * @return array Result array with 'success' and 'message'
     */
    public function deactivate_license() {
        $license_data = $this->get_license_data();

        if (empty($license_data['license_key'])) {
            return array(
                'success' => false,
                'message' => __('No license key found.', 'product-bundle-quantity')
            );
        }

        // Deactivate with API (we don't check the result as we want to clear local data regardless)
        $this->validate_license_with_api($license_data['license_key'], 'deactivate');

        // Clear transient cache
        delete_transient('kbq_license_cache_' . md5($license_data['license_key'] . get_site_url()));

        // Clear license data
        delete_option(self::LICENSE_OPTION);

        // Clear cache
        $this->license_status = null;

        return array(
            'success' => true,
            'message' => __('License deactivated successfully.', 'product-bundle-quantity')
        );
    }

    /**
     * Validate license with API
     *
     * Makes a request to the Code Moksha license API to validate the license key
     *
     * @param string $license_key The license key to validate
     * @param string $action Action type: 'validate', 'activate', 'check', 'deactivate'
     * @return array Result array with 'success', 'message', and optionally 'data'
     */
    private function validate_license_with_api($license_key, $action = 'validate') {
        // Check transient cache for recent validation (only for check/validate actions)
        if (in_array($action, array('validate', 'check'))) {
            $cache_key = 'kbq_license_cache_' . md5($license_key . get_site_url());
            $cached_result = get_transient($cache_key);

            if ($cached_result !== false) {
                return $cached_result;
            }
        }

        // Generate unique device ID based on site URL and server info
        $device_id = $this->generate_device_id();

        // Build request data according to API specification
        $request_data = array(
            'licenseKey' => $license_key,
            'websiteUrl' => get_site_url(),
            'deviceId'   => $device_id,
        );

        // Make API request with JSON body
        $response = wp_remote_post(self::API_ENDPOINT, array(
            'timeout' => 15,
            'sslverify' => true, // Always verify SSL in production
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept'       => 'application/json',
                'User-Agent'   => 'Product-Bundle-Quantity/2.5.0; WordPress/' . get_bloginfo('version'),
            ),
            'body' => json_encode($request_data),
        ));

        // Handle connection errors
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => sprintf(
                    __('API connection failed: %s. Please check your internet connection and try again.', 'product-bundle-quantity'),
                    $response->get_error_message()
                )
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        // Parse JSON response
        $data = json_decode($response_body, true);

        // Handle JSON parsing errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => __('Invalid response from license server. Please try again later.', 'product-bundle-quantity')
            );
        }

        // Handle non-200 HTTP responses
        if ($response_code !== 200) {
            $error_message = isset($data['message']) ? $data['message'] : __('License validation failed.', 'product-bundle-quantity');
            return array(
                'success' => false,
                'message' => $error_message
            );
        }

        // Check if license is valid
        if (!isset($data['valid']) || $data['valid'] !== true) {
            // Handle different error statuses
            $status = isset($data['status']) ? $data['status'] : 'unknown';
            $message = isset($data['message']) ? $data['message'] : __('License validation failed.', 'product-bundle-quantity');

            // Provide user-friendly messages based on status
            switch ($status) {
                case 'expired':
                    $message = __('Your license has expired. Please renew your license to continue using premium features.', 'product-bundle-quantity');
                    if (!empty($data['expirationDate'])) {
                        $message .= ' ' . sprintf(
                            __('Expired on: %s', 'product-bundle-quantity'),
                            date_i18n(get_option('date_format'), strtotime($data['expirationDate']))
                        );
                    }
                    break;

                case 'website_mismatch':
                    $message = __('This license is registered for a different website. Please use the correct license key for this domain or contact support.', 'product-bundle-quantity');
                    break;

                case 'not_found':
                    $message = __('Invalid license key. Please check your license key and try again.', 'product-bundle-quantity');
                    break;

                case 'pending':
                    $message = __('Your payment is still being processed. Please wait a few minutes and try again.', 'product-bundle-quantity');
                    break;

                case 'revoked':
                    $message = __('This license has been revoked. Please contact support for assistance.', 'product-bundle-quantity');
                    break;

                default:
                    // Use the message from the API if available
                    break;
            }

            return array(
                'success' => false,
                'message' => $message,
                'status'  => $status,
            );
        }

        // License is valid - extract data
        $result = array(
            'success' => true,
            'message' => isset($data['message']) ? $data['message'] : __('License is valid and active.', 'product-bundle-quantity'),
            'data' => array(
                'status'          => isset($data['status']) ? $data['status'] : 'active',
                'purchaseDate'    => isset($data['purchaseDate']) ? $data['purchaseDate'] : '',
                'expirationDate'  => isset($data['expirationDate']) ? $data['expirationDate'] : '',
                'features'        => isset($data['features']) ? $data['features'] : array(),
            )
        );

        // Cache successful validation for 12 hours
        if (in_array($action, array('validate', 'check'))) {
            $cache_key = 'kbq_license_cache_' . md5($license_key . get_site_url());
            set_transient($cache_key, $result, self::CACHE_DURATION);
        }

        return $result;
    }

    /**
     * Generate a unique device ID for this WordPress installation
     *
     * @return string Unique device identifier
     */
    private function generate_device_id() {
        $device_id = get_option('kbq_device_id');

        if (empty($device_id)) {
            // Generate a unique ID based on site URL and server info
            $device_id = md5(
                get_site_url() .
                (defined('AUTH_KEY') ? AUTH_KEY : '') .
                (defined('SECURE_AUTH_KEY') ? SECURE_AUTH_KEY : '') .
                $_SERVER['SERVER_ADDR'] ?? ''
            );
            update_option('kbq_device_id', $device_id);
        }

        return $device_id;
    }

    /**
     * Check license validity (daily cron job)
     *
     * This function is called by the daily cron job to verify the license is still valid
     */
    public function check_license_validity() {
        $license_data = $this->get_license_data();

        if (empty($license_data['license_key']) || ($license_data['status'] ?? 'inactive') !== 'active') {
            return;
        }

        // Clear cache before checking to force fresh validation
        delete_transient('kbq_license_cache_' . md5($license_data['license_key'] . get_site_url()));

        // Validate with API
        $validation_result = $this->validate_license_with_api($license_data['license_key'], 'check');

        if (!$validation_result['success']) {
            // License is no longer valid - update status based on error
            $status = isset($validation_result['status']) ? $validation_result['status'] : 'inactive';
            $license_data['status'] = $status;
        } else {
            // License is still valid - update with fresh data from API
            $api_data = $validation_result['data'];
            $license_data['status'] = $api_data['status'] ?? 'active';
            $license_data['expires_at'] = $api_data['expirationDate'] ?? $license_data['expires_at'];
            $license_data['features'] = $api_data['features'] ?? $license_data['features'];
        }

        $license_data['last_checked'] = current_time('mysql');
        update_option(self::LICENSE_OPTION, $license_data);

        // Clear cache
        $this->license_status = null;
    }

    /**
     * Get license data
     *
     * @return array License data with default values if not set
     */
    public function get_license_data() {
        $defaults = array(
            'license_key'   => '',
            'status'        => 'inactive',
            'activated_at'  => '',
            'purchase_date' => '',
            'expires_at'    => '',
            'last_checked'  => '',
            'features'      => array(),
        );

        $saved_data = get_option(self::LICENSE_OPTION, array());

        // Merge saved data with defaults to ensure all keys exist
        return array_merge($defaults, $saved_data);
    }

    /**
     * Check if license is active
     */
    public function is_license_active() {
        if ($this->license_status !== null) {
            return $this->license_status;
        }

        $license_data = $this->get_license_data();

        if (($license_data['status'] ?? 'inactive') !== 'active') {
            $this->license_status = false;
            return false;
        }

        // Check expiration
        if (!empty($license_data['expires_at'])) {
            $expires = strtotime($license_data['expires_at']);
            if ($expires < time()) {
                $this->license_status = false;

                // Update status
                $license_data['status'] = 'expired';
                update_option(self::LICENSE_OPTION, $license_data);

                return false;
            }
        }

        $this->license_status = true;
        return true;
    }

    /**
     * Check if user has access to premium feature
     *
     * @param string|null $feature Feature key to check (e.g., 'custom_colors', 'product_discounts')
     * @return bool True if user has access to the feature
     */
    public function has_premium_access($feature = null) {
        // CRITICAL: First check if license is active
        $is_active = $this->is_license_active();

        // Debug logging (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log(sprintf(
                'KBQ License Check: Feature=%s, License Active=%s',
                $feature ?? 'null',
                $is_active ? 'YES' : 'NO'
            ));
        }

        if (!$is_active) {
            return false;
        }

        // If no specific feature is provided, just check if license is active
        if ($feature === null) {
            return true;
        }

        // Check if the feature exists in our premium features list
        if (!isset(self::PREMIUM_FEATURES[$feature])) {
            // Log unknown feature request
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('KBQ License: Unknown feature requested: ' . $feature);
            }
            return false;
        }

        // Get license data to check features from API
        $license_data = $this->get_license_data();
        $api_features = isset($license_data['features']) && is_array($license_data['features'])
            ? $license_data['features']
            : array();

        // Debug: Log API features
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('KBQ License: API Features = ' . print_r($api_features, true));
        }

        // If no features array from API, grant access to all features (backward compatibility)
        // This ensures that valid licenses without specific feature restrictions get full access
        if (empty($api_features)) {
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('KBQ License: No API features array - granting access (fail-open)');
            }
            return true;
        }

        // Flexible feature matching: Check multiple possible feature name formats
        // This handles variations in how the API might return feature names
        $feature_variations = array(
            $feature,                                    // Exact match (e.g., 'custom_colors')
            str_replace('_', '-', $feature),             // Hyphenated (e.g., 'custom-colors')
            strtolower($feature),                        // Lowercase (e.g., 'custom_colors')
            ucwords(str_replace('_', ' ', $feature)),    // Title case (e.g., 'Custom Colors')
            str_replace('_', '', $feature),              // No separator (e.g., 'customcolors')
        );

        // Additional common API feature name patterns
        $additional_checks = array(
            'custom_colors'      => array('advanced_pricing', 'customization', 'premium_colors', 'color_options'),
            'product_discounts'  => array('bulk_discounts', 'discounts', 'pricing', 'bundle_pricing'),
        );

        // Check all possible variations
        foreach ($feature_variations as $variation) {
            if (in_array($variation, $api_features, true)) {
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    error_log('KBQ License: Feature match found - ' . $variation);
                }
                return true;
            }
        }

        // Check additional known mappings
        if (isset($additional_checks[$feature])) {
            foreach ($additional_checks[$feature] as $possible_name) {
                if (in_array($possible_name, $api_features, true)) {
                    if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                        error_log('KBQ License: Feature mapping match found - ' . $possible_name);
                    }
                    return true;
                }
            }
        }

        // Case-insensitive search as final fallback
        $api_features_lower = array_map('strtolower', $api_features);
        foreach ($feature_variations as $variation) {
            if (in_array(strtolower($variation), $api_features_lower, true)) {
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    error_log('KBQ License: Case-insensitive match found');
                }
                return true;
            }
        }

        // If we still haven't found a match but license is valid and active,
        // grant access anyway (fail-open for valid licenses)
        // This prevents valid paying customers from being locked out due to API inconsistencies
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('KBQ License: No feature match - but granting access (fail-open for valid license)');
        }
        return true;
    }

    /**
     * Mask license key for display
     */
    private function mask_license_key($license_key) {
        if (strlen($license_key) <= 8) {
            return str_repeat('*', strlen($license_key));
        }

        $visible_start = substr($license_key, 0, 4);
        $visible_end = substr($license_key, -4);
        $masked_middle = str_repeat('*', strlen($license_key) - 8);

        return $visible_start . $masked_middle . $visible_end;
    }

    /**
     * Get debug information about license and features
     * Only callable by administrators for troubleshooting
     *
     * @return array Debug information
     */
    public function get_debug_info() {
        if (!current_user_can('manage_options')) {
            return array('error' => 'Insufficient permissions');
        }

        $license_data = $this->get_license_data();
        $is_active = $this->is_license_active();

        return array(
            'license_active' => $is_active,
            'license_status' => $license_data['status'] ?? 'unknown',
            'features_from_api' => $license_data['features'] ?? array(),
            'has_custom_colors_access' => $this->has_premium_access('custom_colors'),
            'has_product_discounts_access' => $this->has_premium_access('product_discounts'),
            'expires_at' => $license_data['expires_at'] ?? 'N/A',
            'last_checked' => $license_data['last_checked'] ?? 'Never',
        );
    }

    /**
     * Display admin notices for license status
     */
    public function display_license_notices() {
        // Only show on plugin pages and product edit pages
        $screen = get_current_screen();
        if (!$screen) {
            return;
        }

        // Show on plugin pages or product edit pages
        $show_notice = (
            strpos($screen->id, 'product-bundle-quantity') !== false ||
            strpos($screen->id, 'kbq-license') !== false ||
            $screen->id === 'product'
        );

        if (!$show_notice) {
            return;
        }

        $license_data = $this->get_license_data();

        // Debug mode notice (only for administrators)
        if (current_user_can('manage_options') && isset($_GET['kbq_debug'])) {
            $debug_info = $this->get_debug_info();
            ?>
            <div class="notice notice-info">
                <p><strong><?php _e('License Debug Information:', 'product-bundle-quantity'); ?></strong></p>
                <pre style="background: #f5f5f5; padding: 10px; overflow: auto;"><?php echo esc_html(print_r($debug_info, true)); ?></pre>
            </div>
            <?php
        }

        // Show notice if license is not active
        if (($license_data['status'] ?? 'inactive') !== 'active') {
            ?>
            <div class="notice notice-info is-dismissible">
                <p>
                    <strong><?php _e('Product Bundle Quantity:', 'product-bundle-quantity'); ?></strong>
                    <?php
                    printf(
                        __('Activate your premium license to unlock all features. %s', 'product-bundle-quantity'),
                        '<a href="' . admin_url('admin.php?page=kbq-license') . '">' . __('Activate Now', 'product-bundle-quantity') . '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
        // Show warning if license is expiring soon
        elseif (!empty($license_data['expires_at'])) {
            $expires = strtotime($license_data['expires_at']);
            $days_remaining = ceil(($expires - time()) / DAY_IN_SECONDS);

            if ($days_remaining > 0 && $days_remaining <= 30) {
                ?>
                <div class="notice notice-warning is-dismissible">
                    <p>
                        <strong><?php _e('Product Bundle Quantity:', 'product-bundle-quantity'); ?></strong>
                        <?php
                        printf(
                            __('Your license will expire in %d days. Please renew to continue using premium features.', 'product-bundle-quantity'),
                            $days_remaining
                        );
                        ?>
                        <a href="https://www.codemoksha.com/product-bundle-quantities" target="_blank"><?php _e('Renew Now', 'product-bundle-quantity'); ?></a>
                    </p>
                </div>
                <?php
            }
        }
    }
}
