<?php
/*
 * Plugin Name: Product Bundle Quantity
 * Description: Create product bundles with automatic image gallery sync, custom quantities, and discounts. Fully customizable colors with theme auto-detection and premium features.
 * Version: 2.5.0
 * Author: Code Moksha
 * Author URI: https://codemoksha.com
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: product-bundle-quantity
 * Domain Path: /languages
 * Requires at least: 5.9
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.5
*/

const CSS_VER = '2.5.0';
const JS_VER = '2.5.0';


if (!defined('ABSPATH')) exit; // Exit if accessed directly

// Load license manager
require_once plugin_dir_path(__FILE__) . 'includes/class-license-manager.php';

class Product_Bundle_Quantity {

    /**
     * License manager instance
     */
    private $license_manager;

    public function __construct() {
        add_action('plugins_loaded', array($this, 'init'));

        // Declare HPOS compatibility
        add_action('before_woocommerce_init', array($this, 'declare_hpos_compatibility'));

        // Initialize license manager
        $this->license_manager = KBQ_License_Manager::get_instance();

        // Run migration for backward compatibility (one-time)
        add_action('admin_init', array($this, 'migrate_old_data'));
    }

    /**
     * Declare compatibility with WooCommerce HPOS
     */
    public function declare_hpos_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        }
    }

    /**
     * Check if user has premium access
     *
     * @param string $feature Feature to check (custom_colors, product_discounts)
     * @return bool
     */
    private function has_premium_access($feature = null) {
        return $this->license_manager->has_premium_access($feature);
    }

    public function init() {
        if (!class_exists('WooCommerce')) return;

        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // Enqueue frontend scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));

        // Add product data tab
        add_filter('woocommerce_product_data_tabs', array($this, 'add_product_data_tab'));
        add_action('woocommerce_product_data_panels', array($this, 'add_product_data_fields'));

        // Save custom fields
        // Priority 20 ensures this runs AFTER WooCommerce's default product meta save (priority 10)
        // This prevents WooCommerce from overwriting our gallery changes
        add_action('woocommerce_process_product_meta', array($this, 'save_bundle_quantity_fields'), 20);

        // ADDITIONAL HOOK: Sync gallery after product is fully saved
        // This hook fires AFTER the product object is completely saved
        add_action('woocommerce_update_product', array($this, 'sync_gallery_after_product_save'), 10, 1);

        // Display bundle options on product page
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_bundle_options'));

        // Adjust cart item price
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_cart_item_data'), 10, 3);
        add_filter('woocommerce_get_cart_item_from_session', array($this, 'get_cart_item_from_session'), 10, 2);
        add_filter('woocommerce_cart_item_price', array($this, 'update_cart_item_price'), 10, 3);


        add_action('wp_ajax_add_bundle_to_cart', array($this, 'add_bundle_to_cart'));
        add_action('wp_ajax_nopriv_add_bundle_to_cart', array($this, 'add_bundle_to_cart'));

        add_action('admin_footer', array($this, 'add_bundle_option_template'));

        add_action('woocommerce_before_calculate_totals', array($this, 'calculate_cart_item_price'), 10, 1);

        add_filter('woocommerce_cart_item_price', array($this, 'update_cart_item_price'), 10, 3);
        add_filter('woocommerce_cart_item_quantity', array($this, 'update_cart_item_quantity'), 10, 3);

        add_action('template_redirect', array($this, 'remove_default_woo_actions'),10);

        // Primary shortcode
        add_shortcode('product_bundle_quantities', array($this, 'bundle_options_shortcode'));

        // Keep old shortcodes for backward compatibility
        add_shortcode('product_bundle_options', array($this, 'bundle_options_shortcode'));
        add_shortcode('kund_bundle_options', array($this, 'bundle_options_shortcode'));

        // Admin menu and settings
        add_action('admin_menu', array($this, 'add_admin_menu'), 9); // Priority 9 to run before license manager
        add_action('admin_init', array($this, 'register_settings'));

        // AJAX handler for theme color detection
        add_action('wp_ajax_kbq_detect_theme_colors', array($this, 'ajax_detect_theme_colors'));

        // AJAX handler for reset colors to default
        add_action('wp_ajax_kbq_reset_colors_to_default', array($this, 'ajax_reset_colors_to_default'));

        // Admin notice for bundle gallery sync confirmation
        add_action('admin_notices', array($this, 'display_gallery_sync_notice'));

    }

    /**
     * Bundle options shortcode handler
     *
     * Displays bundle quantity options for the current product
     *
     * @since 1.0.0
     * @param array $atts Shortcode attributes
     * @return string HTML output of bundle options
     *
     * Usage:
     * [product_bundle_quantities]                    - New primary shortcode (uses current product)
     * [product_bundle_quantities product_id="123"]   - Specify product ID
     * [product_bundle_options]                       - Legacy shortcode (backward compatible)
     * [kund_bundle_options]                          - Legacy shortcode (backward compatible)
     */
    public function bundle_options_shortcode($atts = array()) {
        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'product_id' => 0,
        ), $atts, 'product_bundle_quantities');

        ob_start();
        $this->display_bundle_options($atts['product_id']);
        return ob_get_clean();
    }

    public function enqueue_admin_assets($hook) {
        // Product page assets
        if ('post.php' == $hook && get_post_type() == 'product') {
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');

            wp_enqueue_style('product-bundle-quantity-admin', plugins_url('assets/css/admin.css', __FILE__), array('wp-color-picker'), CSS_VER);
            wp_enqueue_script('product-bundle-quantity-admin', plugins_url('assets/js/admin.js', __FILE__), array('jquery', 'wp-color-picker'), JS_VER, true);
        }

        // Settings page assets
        if ($hook === 'woocommerce_page_product-bundle-quantity-settings') {
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');

            wp_enqueue_style('product-bundle-quantity-settings', plugins_url('assets/css/settings.css', __FILE__), array('wp-color-picker'), CSS_VER);
            wp_enqueue_script('product-bundle-quantity-settings', plugins_url('assets/js/settings.js', __FILE__), array('jquery', 'wp-color-picker'), JS_VER, true);

            wp_localize_script('product-bundle-quantity-settings', 'kbqSettings', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('kbq_detect_colors'),
                'reset_nonce' => wp_create_nonce('kbq_reset_colors'),
                'strings' => array(
                    'detecting' => __('Detecting colors...', 'product-bundle-quantity'),
                    'detected' => __('Theme colors detected and applied!', 'product-bundle-quantity'),
                    'detection_failed' => __('Could not auto-detect theme colors. Using defaults.', 'product-bundle-quantity'),
                    'contrast_warning' => __('Contrast ratio: %s:1. WCAG AA requires 4.5:1 minimum.', 'product-bundle-quantity'),
                    'resetting' => __('Resetting colors...', 'product-bundle-quantity'),
                    'reset_success' => __('Colors have been successfully reset to defaults!', 'product-bundle-quantity'),
                    'reset_failed' => __('Failed to reset colors. Please try again.', 'product-bundle-quantity'),
                    'reset_confirm' => __('Are you sure you want to reset all colors to their default values? This action cannot be undone.', 'product-bundle-quantity'),
                ),
            ));
        }
    }

    public function enqueue_frontend_assets() {
        // Allow shortcodes to trigger asset loading even outside product pages
        if (!is_product() && !$this->should_load_shortcode_assets()) {
            return;
        }

        wp_enqueue_style('product-bundle-quantity-frontend', plugins_url('assets/css/frontend.css', __FILE__), array(), CSS_VER);

        // Inject dynamic CSS variables
        wp_add_inline_style('product-bundle-quantity-frontend', $this->generate_css_variables());

        wp_enqueue_script('product-bundle-quantity-frontend', plugins_url('assets/js/frontend.js', __FILE__), array('jquery'), JS_VER, true);

        wp_localize_script('product-bundle-quantity-frontend',
            'wc_add_to_cart_params',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'cart_url' => wc_get_cart_url()
            )
        );

        // Add WooCommerce currency formatting settings for JavaScript
        wp_localize_script('product-bundle-quantity-frontend',
            'kbq_currency_params',
            array(
                'currency_symbol'          => get_woocommerce_currency_symbol(),
                'currency_position'        => get_option('woocommerce_currency_pos'),
                'thousand_separator'       => wc_get_price_thousand_separator(),
                'decimal_separator'        => wc_get_price_decimal_separator(),
                'number_of_decimals'       => wc_get_price_decimals(),
            )
        );
    }

    /**
     * Check if shortcode assets should be loaded on non-product pages
     *
     * @since 2.5.0
     * @return bool True if any bundle shortcode is present in the content
     */
    private function should_load_shortcode_assets() {
        global $post;

        if (!$post || !isset($post->post_content)) {
            return false;
        }

        // Check if any of our shortcodes exist in the post content
        $shortcodes = array('product_bundle_quantities', 'product_bundle_options', 'kund_bundle_options');
        foreach ($shortcodes as $shortcode) {
            if (has_shortcode($post->post_content, $shortcode)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate CSS custom properties for frontend colors
     *
     * @return string CSS variables
     */
    private function generate_css_variables() {
        $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
        $colors = $settings['colors'];

        // Check for per-product overrides
        global $product;
        if ($product) {
            $product_colors_enabled = get_post_meta($product->get_id(), '_product_bundle_custom_colors_enabled', true);
            if ($product_colors_enabled === 'yes') {
                $product_colors = get_post_meta($product->get_id(), '_product_bundle_custom_colors', true);
                if ($product_colors && is_array($product_colors)) {
                    $colors = array_merge($colors, $product_colors);
                }
            }
        }

        $css = ':root {';
        $css .= '--kbq-selected-border: ' . esc_attr($colors['selected_border']) . ';';
        $css .= '--kbq-selected-bg: ' . esc_attr($colors['selected_bg']) . ';';
        $css .= '--kbq-default-border: ' . esc_attr($colors['default_border']) . ';';
        $css .= '--kbq-button-bg: ' . esc_attr($colors['button_bg']) . ';';
        $css .= '--kbq-button-text: ' . esc_attr($colors['button_text']) . ';';
        $css .= '--kbq-saving-badge: ' . esc_attr($colors['saving_badge']) . ';';
        $css .= '--kbq-price-text: ' . esc_attr($colors['price_text']) . ';';
        $css .= '--kbq-secondary-text: ' . esc_attr($colors['secondary_text']) . ';';
        $css .= '--kbq-muted-text: ' . esc_attr($colors['muted_text']) . ';';
        $css .= '}';

        return $css;
    }

    /**
     * Generate product-specific CSS custom properties
     * Used for shortcode bundles to ensure they get the correct product colors
     *
     * @param int $product_id The product ID to generate CSS variables for
     * @param string $scope_selector Optional CSS selector to scope the variables to
     * @return string CSS variables
     */
    private function generate_product_specific_css_variables($product_id, $scope_selector = ':root') {
        $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
        $colors = $settings['colors'];

        // Check for per-product color overrides
        $product_colors_enabled = get_post_meta($product_id, '_product_bundle_custom_colors_enabled', true);
        if ($product_colors_enabled === 'yes') {
            $product_colors = get_post_meta($product_id, '_product_bundle_custom_colors', true);
            if ($product_colors && is_array($product_colors)) {
                $colors = array_merge($colors, $product_colors);
            }
        }

        $css = $scope_selector . ' {';
        $css .= '--kbq-selected-border: ' . esc_attr($colors['selected_border']) . ';';
        $css .= '--kbq-selected-bg: ' . esc_attr($colors['selected_bg']) . ';';
        $css .= '--kbq-default-border: ' . esc_attr($colors['default_border']) . ';';
        $css .= '--kbq-button-bg: ' . esc_attr($colors['button_bg']) . ';';
        $css .= '--kbq-button-text: ' . esc_attr($colors['button_text']) . ';';
        $css .= '--kbq-saving-badge: ' . esc_attr($colors['saving_badge']) . ';';
        $css .= '--kbq-price-text: ' . esc_attr($colors['price_text']) . ';';
        $css .= '--kbq-secondary-text: ' . esc_attr($colors['secondary_text']) . ';';
        $css .= '--kbq-muted-text: ' . esc_attr($colors['muted_text']) . ';';
        $css .= '}';

        return $css;
    }

    public function add_product_data_tab($tabs) {
        $tabs['bundle_quantity'] = array(
            'label' => __('Bundle Quantities', 'product-bundle-quantity'),
            'target' => 'bundle_quantity_options',
            'class' => array('show_if_simple', 'show_if_variable'),
        );
        return $tabs;
    }



public function add_product_data_fields() {
    global $post;

    $has_discount_access = $this->has_premium_access('product_discounts');
    ?>
    <div id='bundle_quantity_options' class='panel woocommerce_options_panel'>
        <div class='options_group'>
            <?php
            woocommerce_wp_checkbox(array(
                'id' => '_enable_bundle_quantity',
                'label' => __('Enable Bundle Quantity', 'product-bundle-quantity'),
                'description' => __('Enable bundle quantity options for this product', 'product-bundle-quantity')
            ));
            ?>

            <div id="kbq_product_color_override" style="<?php echo get_post_meta($post->ID, '_enable_bundle_quantity', true) === 'yes' ? '' : 'display:none;'; ?>">
                <?php
                $has_custom_colors_access = $this->has_premium_access('custom_colors');

                if (!$has_custom_colors_access): ?>
                    <div class="kbq-premium-notice" style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; padding: 15px; margin: 10px 0;">
                        <p style="margin: 0; display: flex; align-items: center; gap: 10px;">
                            <span class="dashicons dashicons-lock" style="color: #856404;"></span>
                            <strong><?php _e('Premium Feature:', 'product-bundle-quantity'); ?></strong>
                            <?php _e('Custom colors are available with a premium license.', 'product-bundle-quantity'); ?>
                            <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>" class="button button-small button-primary" style="margin-left: auto;">
                                <?php _e('Upgrade Now', 'product-bundle-quantity'); ?>
                            </a>
                        </p>
                    </div>
                <?php endif;

                woocommerce_wp_checkbox(array(
                    'id' => '_product_bundle_custom_colors_enabled',
                    'label' => __('Use Custom Colors', 'product-bundle-quantity'),
                    'description' => __('Override global color settings for this product', 'product-bundle-quantity') . ($has_custom_colors_access ? '' : ' ' . __('(Requires Premium License)', 'product-bundle-quantity')),
                    'custom_attributes' => !$has_custom_colors_access ? array('disabled' => 'disabled') : array()
                ));
                ?>

                <div id="kbq_custom_colors_fields" style="<?php echo get_post_meta($post->ID, '_product_bundle_custom_colors_enabled', true) === 'yes' ? '' : 'display:none;'; ?>">
                    <?php
                    $custom_colors = get_post_meta($post->ID, '_product_bundle_custom_colors', true);
                    $global_settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
                    $default_colors = $global_settings['colors'];

                    $color_fields = array(
                        'selected_border' => __('Selected Bundle Border', 'product-bundle-quantity'),
                        'selected_bg' => __('Selected Bundle Background', 'product-bundle-quantity'),
                        'button_bg' => __('Add to Cart Button', 'product-bundle-quantity'),
                        'saving_badge' => __('Saving Badge', 'product-bundle-quantity'),
                    );

                    foreach ($color_fields as $key => $label):
                        $value = isset($custom_colors[$key]) ? $custom_colors[$key] : $default_colors[$key];
                    ?>
                        <p class="form-field">
                            <label for="_product_bundle_color_<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?></label>
                            <input type="text"
                                   id="_product_bundle_color_<?php echo esc_attr($key); ?>"
                                   name="_product_bundle_custom_colors[<?php echo esc_attr($key); ?>]"
                                   value="<?php echo esc_attr($value); ?>"
                                   class="kbq-product-color-picker"
                                   data-default-color="<?php echo esc_attr($default_colors[$key]); ?>" />
                            <span class="description"><?php printf(__('Global default: %s', 'product-bundle-quantity'), esc_html($default_colors[$key])); ?></span>
                        </p>
                    <?php endforeach; ?>

                    <p class="form-field">
                        <label>&nbsp;</label>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=product-bundle-quantity-settings')); ?>" class="button button-secondary" target="_blank">
                            <?php _e('Manage Global Settings', 'product-bundle-quantity'); ?>
                        </a>
                    </p>
                </div>
            </div>

            <?php if (!$has_discount_access): ?>
                <div class="kbq-premium-notice" style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; padding: 15px; margin: 10px 0;">
                    <p style="margin: 0; display: flex; align-items: center; gap: 10px;">
                        <span class="dashicons dashicons-lock" style="color: #856404;"></span>
                        <strong><?php _e('Premium Feature:', 'product-bundle-quantity'); ?></strong>
                        <?php _e('Product discounts are available with a premium license. Upgrade to set custom discount percentages.', 'product-bundle-quantity'); ?>
                        <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>" class="button button-small button-primary" style="margin-left: auto;">
                            <?php _e('Upgrade Now', 'product-bundle-quantity'); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>

            <div id="bundle_quantity_options_container">
                <?php
                $bundle_options = get_post_meta($post->ID, '_bundle_options', true);
                if ($bundle_options) {
                    foreach ($bundle_options as $index => $option) {
                        ?>
                        <div class="bundle-option" data-index="<?php echo $index; ?>">
                            <p class="form-field">
                                <label><?php _e('Quantity', 'product-bundle-quantity'); ?></label>
                                <input type="number" name="bundle_options[<?php echo $index; ?>][quantity]" value="<?php echo esc_attr($option['quantity']); ?>" min="1" step="1" />
                            </p>
                            <p class="form-field">
                                <label><?php _e('Discount (%)', 'product-bundle-quantity'); ?><?php if (!$has_discount_access) echo ' <span style="color: #856404;">(' . __('Premium', 'product-bundle-quantity') . ')</span>'; ?></label>
                                <input type="number"
                                       name="bundle_options[<?php echo $index; ?>][discount]"
                                       value="<?php echo esc_attr($option['discount']); ?>"
                                       min="0"
                                       max="100"
                                       step="0.01"
                                       <?php if (!$has_discount_access) echo 'disabled="disabled" readonly'; ?>
                                       class="kbq-discount-field" />
                                <?php if (!$has_discount_access): ?>
                                    <input type="hidden" name="bundle_options[<?php echo $index; ?>][discount]" value="0" />
                                <?php endif; ?>
                            </p>
                            <p class="form-field">
                                <label><?php _e('Image', 'product-bundle-quantity'); ?></label>
                                <input type="hidden" class="bundle_image_id" name="bundle_options[<?php echo $index; ?>][image]" value="<?php echo esc_attr($option['image']); ?>" />
                                <img src="<?php echo wp_get_attachment_url($option['image']); ?>" alt="" class="bundle-image-preview" style="max-width: 100px; max-height: 100px; <?php echo $option['image'] ? '' : 'display: none;'; ?>" />
                                <button type="button" class="button upload_image_button" data-index="<?php echo $index; ?>"><?php _e('Upload/Add Image', 'product-bundle-quantity'); ?></button>
                                <button type="button" class="button remove_image_button" data-index="<?php echo $index; ?>" <?php echo $option['image'] ? '' : 'style="display:none;"'; ?>><?php _e('Remove Image', 'product-bundle-quantity'); ?></button>
                            </p>
                            <button type="button" class="button remove-bundle-option"><?php _e('Remove', 'product-bundle-quantity'); ?></button>
                        </div>
                        <?php
                    }
                }
                ?>
            </div>
            <button type="button" id="add_bundle_option" class="button"><?php _e('Add Bundle Option', 'product-bundle-quantity'); ?></button>
        </div>
    </div>
    <?php
}


    public function save_bundle_quantity_fields($post_id) {
        $enable_bundle_quantity = isset($_POST['_enable_bundle_quantity']) ? 'yes' : 'no';
        update_post_meta($post_id, '_enable_bundle_quantity', $enable_bundle_quantity);

        // Save bundle options with premium validation
        $bundle_options = isset($_POST['bundle_options']) ? $_POST['bundle_options'] : array();
        $has_discount_access = $this->has_premium_access('product_discounts');

        // Sanitize and validate bundle options
        if (!empty($bundle_options)) {
            foreach ($bundle_options as &$option) {
                // Enforce discount restrictions for non-premium users
                if (!$has_discount_access) {
                    $option['discount'] = ''; // Set to empty string for non-premium users
                } else {
                    // For premium users, ensure discount is a clean numeric value or empty
                    $discount_value = isset($option['discount']) ? floatval($option['discount']) : 0;
                    // If discount is less than 0.01%, treat it as no discount
                    $option['discount'] = ($discount_value >= 0.01) ? $discount_value : '';
                }
            }
            unset($option); // Break the reference
        }

        update_post_meta($post_id, '_bundle_options', $bundle_options);

        // Automatically add bundle images to product gallery (initial save)
        if ($enable_bundle_quantity === 'yes' && !empty($bundle_options)) {
            $this->add_bundle_images_to_gallery($post_id, $bundle_options);
        }

        // Save custom color override settings with premium validation
        $has_custom_colors_access = $this->has_premium_access('custom_colors');
        $custom_colors_enabled = isset($_POST['_product_bundle_custom_colors_enabled']) ? 'yes' : 'no';

        // Only allow custom colors if user has premium access
        if ($has_custom_colors_access && $custom_colors_enabled === 'yes' && isset($_POST['_product_bundle_custom_colors'])) {
            update_post_meta($post_id, '_product_bundle_custom_colors_enabled', $custom_colors_enabled);

            $custom_colors = array();
            foreach ($_POST['_product_bundle_custom_colors'] as $key => $value) {
                $custom_colors[$key] = sanitize_hex_color($value);
            }
            update_post_meta($post_id, '_product_bundle_custom_colors', $custom_colors);
        } else {
            // Disable custom colors for non-premium users
            update_post_meta($post_id, '_product_bundle_custom_colors_enabled', 'no');
            delete_post_meta($post_id, '_product_bundle_custom_colors');
        }
    }

    /**
     * Display admin notice when bundle images are synced to gallery
     *
     * @since 2.5.0
     * @return void
     */
    public function display_gallery_sync_notice() {
        // Only show on product edit page
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'product') {
            return;
        }

        // Check if we have a success transient
        global $post;
        if (!$post) {
            return;
        }

        $transient_key = 'kbq_gallery_sync_success_' . $post->ID;
        if (get_transient($transient_key)) {
            // Delete transient so it only shows once
            delete_transient($transient_key);

            // Get bundle options to count images
            $bundle_options = get_post_meta($post->ID, '_bundle_options', true);
            $image_count = 0;
            if (is_array($bundle_options)) {
                foreach ($bundle_options as $option) {
                    if (!empty($option['image'])) {
                        $image_count++;
                    }
                }
            }

            if ($image_count > 0) {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p>
                        <strong><?php _e('Bundle Images Synced:', 'product-bundle-quantity'); ?></strong>
                        <?php
                        printf(
                            _n(
                                '%d bundle image has been automatically added to the product gallery.',
                                '%d bundle images have been automatically added to the product gallery.',
                                $image_count,
                                'product-bundle-quantity'
                            ),
                            $image_count
                        );
                        ?>
                    </p>
                </div>
                <?php
            }
        }
    }

    /**
     * Add bundle images to product gallery
     *
     * Automatically adds all bundle images to the WooCommerce product gallery
     * using WordPress/WooCommerce best practices with proper hooks, validation,
     * and error handling.
     *
     * @since 2.5.0
     * @param int   $post_id        Product ID
     * @param array $bundle_options Array of bundle options with image IDs
     * @return bool True if gallery was updated, false otherwise
     *
     * @hook kbq_before_add_bundle_images_to_gallery Fires before images are added to gallery
     * @hook kbq_bundle_image_ids Filter to modify bundle image IDs before adding
     * @hook kbq_after_add_bundle_images_to_gallery Fires after images are added to gallery
     */
    private function add_bundle_images_to_gallery($post_id, $bundle_options) {
        // Validate inputs
        if (empty($post_id) || empty($bundle_options) || !is_array($bundle_options)) {
            return false;
        }

        /**
         * Fires before bundle images are added to product gallery
         *
         * @since 2.5.0
         * @param int   $post_id        Product ID
         * @param array $bundle_options Bundle options array
         */
        do_action('kbq_before_add_bundle_images_to_gallery', $post_id, $bundle_options);

        // Get current product gallery directly from meta
        // Using direct meta access because we're inside woocommerce_process_product_meta hook
        $gallery = get_post_meta($post_id, '_product_image_gallery', true);
        $gallery_ids = !empty($gallery) ? array_map('intval', explode(',', $gallery)) : array();

        // Collect all bundle image IDs with validation
        $bundle_image_ids = array();
        foreach ($bundle_options as $option) {
            if (!empty($option['image']) && is_numeric($option['image'])) {
                $image_id = intval($option['image']);
                // Validate that image attachment exists and is an image
                if ($this->validate_image_attachment($image_id)) {
                    $bundle_image_ids[] = $image_id;
                }
            }
        }

        // Remove duplicates from bundle images
        $bundle_image_ids = array_unique($bundle_image_ids);

        /**
         * Filter bundle image IDs before adding to gallery
         *
         * Allows developers to modify, add, or remove image IDs before they're
         * added to the product gallery.
         *
         * @since 2.5.0
         * @param array $bundle_image_ids Array of image attachment IDs
         * @param int   $post_id          Product ID
         * @param array $bundle_options   Bundle options array
         */
        $bundle_image_ids = apply_filters('kbq_bundle_image_ids', $bundle_image_ids, $post_id, $bundle_options);

        // Add bundle images to gallery if they're not already there
        $added_images = array();
        foreach ($bundle_image_ids as $image_id) {
            if (!in_array($image_id, $gallery_ids)) {
                $gallery_ids[] = $image_id;
                $added_images[] = $image_id;
            }
        }

        // Only update if changes were made
        if (!empty($added_images)) {
            // Remove empty values and ensure all are numeric (safety check)
            $gallery_ids = array_filter($gallery_ids, function($id) {
                return !empty($id) && is_numeric($id);
            });

            // Reset array keys to prevent gaps
            $gallery_ids = array_values($gallery_ids);

            // Convert to comma-separated string
            $new_gallery = implode(',', $gallery_ids);

            // Update gallery using direct meta update
            // Using update_post_meta because we're inside woocommerce_process_product_meta hook
            // This prevents conflicts with WooCommerce's save process
            update_post_meta($post_id, '_product_image_gallery', $new_gallery);

            /**
             * Fires after bundle images are added to product gallery
             *
             * @since 2.5.0
             * @param int   $post_id      Product ID
             * @param array $added_images Array of image IDs that were added
             * @param array $gallery_ids  Complete array of gallery image IDs
             */
            do_action('kbq_after_add_bundle_images_to_gallery', $post_id, $added_images, $gallery_ids);

            return true;
        }

        return false;
    }

    /**
     * Validate image attachment
     *
     * Validates that an attachment ID is a valid image file
     *
     * @since 2.5.0
     * @param int $attachment_id Attachment ID to validate
     * @return bool True if valid image attachment, false otherwise
     */
    private function validate_image_attachment($attachment_id) {
        // Check if attachment exists
        if (get_post_type($attachment_id) !== 'attachment') {
            return false;
        }

        // Check if it's an image mime type
        $mime_type = get_post_mime_type($attachment_id);
        $allowed_mime_types = array(
            'image/jpeg',
            'image/jpg',
            'image/png',
            'image/gif',
            'image/webp',
            'image/svg+xml'
        );

        /**
         * Filter allowed image mime types for bundle images
         *
         * @since 2.5.0
         * @param array $allowed_mime_types Array of allowed mime types
         * @param int   $attachment_id      Attachment ID being validated
         */
        $allowed_mime_types = apply_filters('kbq_bundle_image_allowed_mime_types', $allowed_mime_types, $attachment_id);

        if (!in_array($mime_type, $allowed_mime_types, true)) {
            return false;
        }

        return true;
    }

    /**
     * Sync bundle images to product gallery after product is fully saved
     *
     * This method runs on the woocommerce_update_product hook, which fires AFTER
     * the product has been completely saved. This ensures our gallery changes
     * persist and aren't overwritten by WooCommerce.
     *
     * @since 2.5.0
     * @param int $product_id Product ID
     * @return void
     */
    public function sync_gallery_after_product_save($product_id) {
        // Get product object
        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }

        // Check if bundle quantities are enabled
        $enable_bundle = get_post_meta($product_id, '_enable_bundle_quantity', true);
        if ($enable_bundle !== 'yes') {
            return;
        }

        // Get bundle options
        $bundle_options = get_post_meta($product_id, '_bundle_options', true);
        if (empty($bundle_options) || !is_array($bundle_options)) {
            return;
        }

        // Collect bundle image IDs
        $bundle_image_ids = array();
        foreach ($bundle_options as $option) {
            if (!empty($option['image']) && is_numeric($option['image'])) {
                $image_id = intval($option['image']);
                if ($this->validate_image_attachment($image_id)) {
                    $bundle_image_ids[] = $image_id;
                }
            }
        }

        if (empty($bundle_image_ids)) {
            return;
        }

        // Remove duplicates
        $bundle_image_ids = array_unique($bundle_image_ids);

        // Get current gallery
        $gallery_ids = $product->get_gallery_image_ids();

        // Add bundle images if not already in gallery
        $added = false;
        foreach ($bundle_image_ids as $image_id) {
            if (!in_array($image_id, $gallery_ids)) {
                $gallery_ids[] = $image_id;
                $added = true;
            }
        }

        // Update gallery if changes were made
        if ($added) {
            $product->set_gallery_image_ids($gallery_ids);
            $product->save();

            // Log success
            error_log('KBQ AFTER-SAVE: Successfully added bundle images to gallery. Product ID: ' . $product_id . ', Gallery IDs: ' . implode(',', $gallery_ids));

            // Set transient for admin notice
            set_transient('kbq_gallery_sync_success_' . $product_id, true, 30);
        }
    }

    /**
     * Get default plugin settings
     *
     * @return array Default settings with colors and display options
     */
    private function get_default_settings() {
        return array(
            'colors' => array(
                'selected_border'   => '#f7dc5a',
                'selected_bg'       => '#fff1a4',
                'default_border'    => '#e0e0e0',
                'button_bg'         => '#ffc00b',
                'button_text'       => '#ffffff',
                'saving_badge'      => '#ff0000',
                'price_text'        => '#333333',
                'secondary_text'    => '#666666',
                'muted_text'        => '#999999',
            ),
            'display' => array(
                'show_stock_label'     => true,
                'show_security_message' => true,
            ),
            'theme_colors_detected' => false,
            'last_theme'            => '',
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting(
            'product_bundle_quantity_settings_group',
            'product_bundle_quantity_settings',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_settings'),
                'default' => $this->get_default_settings(),
            )
        );

        // Handle reset colors action
        if (isset($_GET['kbq_action']) && $_GET['kbq_action'] === 'reset_colors' && isset($_GET['_wpnonce'])) {
            if (wp_verify_nonce($_GET['_wpnonce'], 'kbq_reset_colors_action')) {
                $this->reset_colors_to_default();
                wp_redirect(add_query_arg(array('page' => 'product-bundle-quantity-settings', 'tab' => 'colors', 'kbq_reset' => 'success'), admin_url('admin.php')));
                exit;
            }
        }
    }

    /**
     * Reset colors to default (non-AJAX version)
     */
    private function reset_colors_to_default() {
        $settings = get_option('kund_bundle_quantity_settings', $this->get_default_settings());
        $default_settings = $this->get_default_settings();

        $settings['colors'] = $default_settings['colors'];
        $settings['theme_colors_detected'] = false;
        $settings['last_theme'] = '';

        update_option('kund_bundle_quantity_settings', $settings);
    }

    /**
     * Sanitize and validate settings
     *
     * @param array $input Raw input from settings form
     * @return array Sanitized settings
     */
    public function sanitize_settings($input) {
        // Get current settings to preserve values
        $current_settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
        $sanitized = $current_settings;

        // Sanitize colors (validate hex format)
        if (isset($input['colors']) && is_array($input['colors'])) {
            foreach ($input['colors'] as $key => $color) {
                $sanitized_color = sanitize_hex_color($color);
                if ($sanitized_color) {
                    $sanitized['colors'][$key] = $sanitized_color;
                } else {
                    // Use default if invalid
                    $defaults = $this->get_default_settings();
                    $sanitized['colors'][$key] = isset($defaults['colors'][$key]) ? $defaults['colors'][$key] : '#000000';
                }

                // Validate color contrast for accessibility
                if ($key === 'button_text' && isset($input['colors']['button_bg'])) {
                    $contrast_ratio = $this->calculate_contrast_ratio(
                        $sanitized['colors'][$key],
                        sanitize_hex_color($input['colors']['button_bg'])
                    );

                    if ($contrast_ratio < 4.5) {
                        add_settings_error(
                            'product_bundle_quantity_settings',
                            'contrast_ratio',
                            sprintf(
                                __('Warning: The contrast ratio between button text and background is %s:1. WCAG 2.1 AA requires at least 4.5:1 for accessibility.', 'product-bundle-quantity'),
                                number_format($contrast_ratio, 2)
                            ),
                            'warning'
                        );
                    }
                }
            }
        }

        // Sanitize display toggles
        // Important: Checkboxes only send data when checked, so we need to explicitly handle unchecked state
        if (!isset($sanitized['display'])) {
            $sanitized['display'] = array();
        }

        // Check if we're saving from the display tab using the hidden marker field
        // This ensures we only update display settings when they're actually being saved
        if (isset($input['display']['_tab_marker'])) {
            // We're saving from display tab - update checkbox values
            // Checkboxes only send value when checked, so !empty() returns true for checked, false for unchecked
            $sanitized['display']['show_stock_label'] = isset($input['display']['show_stock_label']) && $input['display']['show_stock_label'] === '1';
            $sanitized['display']['show_security_message'] = isset($input['display']['show_security_message']) && $input['display']['show_security_message'] === '1';
        }

        // Preserve theme detection metadata
        $sanitized['theme_colors_detected'] = isset($input['theme_colors_detected']) ? (bool)$input['theme_colors_detected'] : $current_settings['theme_colors_detected'];
        $sanitized['last_theme'] = isset($input['last_theme']) ? sanitize_text_field($input['last_theme']) : $current_settings['last_theme'];

        return $sanitized;
    }

    /**
     * Calculate contrast ratio between two colors (WCAG 2.1)
     *
     * @param string $color1 Hex color code
     * @param string $color2 Hex color code
     * @return float Contrast ratio
     */
    private function calculate_contrast_ratio($color1, $color2) {
        $l1 = $this->get_relative_luminance($color1);
        $l2 = $this->get_relative_luminance($color2);

        $lighter = max($l1, $l2);
        $darker = min($l1, $l2);

        return ($lighter + 0.05) / ($darker + 0.05);
    }

    /**
     * Get relative luminance of a color
     *
     * @param string $hex Hex color code
     * @return float Relative luminance value
     */
    private function get_relative_luminance($hex) {
        $hex = ltrim($hex, '#');
        $rgb = array_map('hexdec', str_split($hex, 2));

        $rgb = array_map(function($v) {
            $v = $v / 255;
            return $v <= 0.03928 ? $v / 12.92 : pow(($v + 0.055) / 1.055, 2.4);
        }, $rgb);

        return 0.2126 * $rgb[0] + 0.7152 * $rgb[1] + 0.0722 * $rgb[2];
    }

    /**
     * Lighten a color by a percentage
     *
     * @param string $hex Hex color code
     * @param int $percent Percentage to lighten (0-100)
     * @return string Lightened hex color code
     */
    private function lighten_color($hex, $percent) {
        $hex = ltrim($hex, '#');
        $rgb = array_map('hexdec', str_split($hex, 2));

        foreach ($rgb as &$value) {
            $value = min(255, $value + (255 - $value) * ($percent / 100));
        }

        return '#' . implode('', array_map(function($v) {
            return str_pad(dechex(round($v)), 2, '0', STR_PAD_LEFT);
        }, $rgb));
    }

    /**
     * Detect and apply theme colors
     *
     * @return bool True if colors were detected, false otherwise
     */
    public function detect_theme_colors() {
        $detected_colors = array();
        $current_theme = wp_get_theme()->get_stylesheet();

        // Method 1: theme.json (Block Themes - WordPress 5.9+)
        if (function_exists('wp_get_global_settings')) {
            $theme_json_colors = $this->extract_theme_json_colors();
            if (!empty($theme_json_colors)) {
                $detected_colors = array_merge($detected_colors, $theme_json_colors);
            }
        }

        // Method 2: get_theme_mod() (Classic Themes)
        $theme_mod_colors = $this->extract_theme_mod_colors();
        if (!empty($theme_mod_colors)) {
            $detected_colors = array_merge($detected_colors, $theme_mod_colors);
        }

        // Apply detected colors to settings
        if (!empty($detected_colors)) {
            $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
            $mapped_colors = $this->map_detected_colors($detected_colors);

            // Merge mapped colors with existing colors
            $settings['colors'] = array_merge($settings['colors'], $mapped_colors);
            $settings['theme_colors_detected'] = true;
            $settings['last_theme'] = $current_theme;

            update_option('product_bundle_quantity_settings', $settings);

            return true;
        }

        return false;
    }

    /**
     * Extract colors from theme.json
     *
     * @return array Detected colors
     */
    private function extract_theme_json_colors() {
        $colors = array();
        $theme_json_path = get_stylesheet_directory() . '/theme.json';

        if (!file_exists($theme_json_path)) {
            return $colors;
        }

        $theme_json_content = file_get_contents($theme_json_path);
        $theme_json = json_decode($theme_json_content, true);

        if (isset($theme_json['settings']['color']['palette'])) {
            foreach ($theme_json['settings']['color']['palette'] as $color) {
                // Map theme color slugs to our settings
                $slug = strtolower($color['slug']);
                if (in_array($slug, array('primary', 'accent', 'base', 'contrast', 'secondary'))) {
                    $colors[$slug] = $color['color'];
                }
            }
        }

        return $colors;
    }

    /**
     * Extract colors from theme mods
     *
     * @return array Detected colors
     */
    private function extract_theme_mod_colors() {
        $colors = array();

        // Common theme mod keys
        $theme_mod_keys = array(
            'primary_color',
            'accent_color',
            'header_textcolor',
            'background_color',
            'link_color',
        );

        foreach ($theme_mod_keys as $key) {
            $value = get_theme_mod($key);
            if ($value) {
                // Ensure it's a valid hex color
                $colors[$key] = '#' . ltrim($value, '#');
            }
        }

        return $colors;
    }

    /**
     * Map detected colors to plugin settings
     *
     * @param array $detected Detected colors from theme
     * @return array Mapped colors for plugin settings
     */
    private function map_detected_colors($detected) {
        $mapping = array();

        // Map detected colors to plugin color settings
        if (isset($detected['primary']) || isset($detected['primary_color'])) {
            $primary = isset($detected['primary']) ? $detected['primary'] : $detected['primary_color'];
            $mapping['selected_border'] = $primary;
            $mapping['button_bg'] = $primary;
        }

        if (isset($detected['accent']) || isset($detected['accent_color'])) {
            $accent = isset($detected['accent']) ? $detected['accent'] : $detected['accent_color'];
            $mapping['saving_badge'] = $accent;
        }

        if (isset($detected['base'])) {
            $mapping['selected_bg'] = $this->lighten_color($detected['base'], 40);
        }

        if (isset($detected['contrast'])) {
            $mapping['price_text'] = $detected['contrast'];
        }

        if (isset($detected['link_color'])) {
            $mapping['button_bg'] = $detected['link_color'];
        }

        return $mapping;
    }

    /**
     * AJAX handler for theme color detection
     */
    public function ajax_detect_theme_colors() {
        check_ajax_referer('kbq_detect_colors', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Insufficient permissions', 'product-bundle-quantity'));
        }

        $detected = $this->detect_theme_colors();

        if ($detected) {
            $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
            wp_send_json_success(array(
                'colors' => $settings['colors'],
                'message' => __('Theme colors detected successfully!', 'product-bundle-quantity')
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('Could not auto-detect theme colors. Using defaults.', 'product-bundle-quantity')
            ));
        }
    }

    /**
     * AJAX handler for resetting colors to default
     */
    public function ajax_reset_colors_to_default() {
        check_ajax_referer('kbq_reset_colors', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Insufficient permissions', 'product-bundle-quantity'));
        }

        // Get current settings
        $settings = get_option('kund_bundle_quantity_settings', $this->get_default_settings());

        // Get default colors
        $default_settings = $this->get_default_settings();

        // Reset only the colors to default, preserve other settings
        $settings['colors'] = $default_settings['colors'];
        $settings['theme_colors_detected'] = false;
        $settings['last_theme'] = '';

        // Update the settings
        $updated = update_option('kund_bundle_quantity_settings', $settings);

        if ($updated || get_option('kund_bundle_quantity_settings') == $settings) {
            wp_send_json_success(array(
                'colors' => $settings['colors'],
                'message' => __('Colors have been successfully reset to defaults!', 'product-bundle-quantity')
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('Failed to reset colors. Please try again.', 'product-bundle-quantity')
            ));
        }
    }

    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        // Main menu page - Dashboard
        add_menu_page(
            __('Bundle Quantities', 'product-bundle-quantity'),           // Page title
            __('Bundle Quantities', 'product-bundle-quantity'),           // Menu title
            'manage_woocommerce',                                         // Capability
            'product-bundle-quantity',                                     // Menu slug
            array($this, 'render_dashboard_page'),                        // Callback
            'dashicons-products',                                          // Icon
            56                                                             // Position (after WooCommerce)
        );

        // Dashboard submenu (same as parent)
        add_submenu_page(
            'product-bundle-quantity',                                     // Parent slug
            __('Dashboard', 'product-bundle-quantity'),                   // Page title
            __('Dashboard', 'product-bundle-quantity'),                   // Menu title
            'manage_woocommerce',                                         // Capability
            'product-bundle-quantity',                                     // Menu slug (same as parent)
            array($this, 'render_dashboard_page')                         // Callback
        );

        // Settings submenu
        add_submenu_page(
            'product-bundle-quantity',                                     // Parent slug
            __('Bundle Quantities Settings', 'product-bundle-quantity'),  // Page title
            __('Settings', 'product-bundle-quantity'),                    // Menu title
            'manage_woocommerce',                                         // Capability
            'product-bundle-quantity-settings',                           // Menu slug
            array($this, 'render_settings_page')                          // Callback
        );

        // License submenu will be added by License Manager class
    }

    /**
     * Render the dashboard page
     */
    public function render_dashboard_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'product-bundle-quantity'));
        }

        // Get license status
        $is_premium = $this->has_premium_access();

        // Get plugin statistics
        global $wpdb;
        $products_with_bundles = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_id)
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_enable_bundle_quantity'
            AND meta_value = 'yes'
        ");

        ?>
        <div class="wrap">
            <h1><?php _e('Bundle Quantities Dashboard', 'product-bundle-quantity'); ?></h1>

            <!-- License Status Banner -->
            <?php if (!$is_premium): ?>
                <div class="notice notice-warning" style="border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; background: #fff;">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span class="dashicons dashicons-lock" style="font-size: 48px; color: #ffc107;"></span>
                        <div style="flex: 1;">
                            <h2 style="margin: 0 0 10px 0;"><?php _e('Unlock Premium Features', 'product-bundle-quantity'); ?></h2>
                            <p style="margin: 0 0 10px 0; font-size: 14px;">
                                <?php _e('Get access to custom colors, product discounts, and priority support.', 'product-bundle-quantity'); ?>
                            </p>
                            <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>" class="button button-primary">
                                <?php _e('Activate License', 'product-bundle-quantity'); ?>
                            </a>
                            <a href="https://www.codemoksha.com/product-bundle-quantities" target="_blank" class="button button-secondary">
                                <?php _e('Get Premium', 'product-bundle-quantity'); ?>
                            </a>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="notice notice-success" style="border-left: 4px solid #46b450; padding: 15px; margin: 20px 0; background: #fff;">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span class="dashicons dashicons-yes-alt" style="font-size: 48px; color: #46b450;"></span>
                        <div>
                            <h2 style="margin: 0 0 5px 0;"><?php _e('Premium Active', 'product-bundle-quantity'); ?></h2>
                            <p style="margin: 0; font-size: 14px;">
                                <?php _e('You have access to all premium features.', 'product-bundle-quantity'); ?>
                                <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>"><?php _e('Manage License', 'product-bundle-quantity'); ?></a>
                            </p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0;">
                <!-- Products with Bundles -->
                <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span class="dashicons dashicons-products" style="font-size: 48px; color: #2271b1;"></span>
                        <div>
                            <h3 style="margin: 0; font-size: 32px; color: #2271b1;"><?php echo number_format($products_with_bundles); ?></h3>
                            <p style="margin: 5px 0 0 0; color: #646970;"><?php _e('Products with Bundles', 'product-bundle-quantity'); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Premium Features -->
                <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span class="dashicons dashicons-star-filled" style="font-size: 48px; color: #ffc107;"></span>
                        <div>
                            <h3 style="margin: 0; font-size: 32px; color: #ffc107;"><?php echo $is_premium ? '2/2' : '0/2'; ?></h3>
                            <p style="margin: 5px 0 0 0; color: #646970;"><?php _e('Premium Features Active', 'product-bundle-quantity'); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Plugin Version -->
                <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span class="dashicons dashicons-admin-plugins" style="font-size: 48px; color: #46b450;"></span>
                        <div>
                            <h3 style="margin: 0; font-size: 32px; color: #46b450;">2.5.0</h3>
                            <p style="margin: 5px 0 0 0; color: #646970;"><?php _e('Plugin Version', 'product-bundle-quantity'); ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin: 20px 0; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                <h2 style="margin: 0 0 20px 0;"><?php _e('Quick Actions', 'product-bundle-quantity'); ?></h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                    <a href="<?php echo admin_url('post-new.php?post_type=product'); ?>" class="button button-secondary" style="height: auto; padding: 15px; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 10px; text-decoration: none;">
                        <span class="dashicons dashicons-plus-alt" style="font-size: 32px;"></span>
                        <?php _e('Add New Product', 'product-bundle-quantity'); ?>
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=product-bundle-quantity-settings'); ?>" class="button button-secondary" style="height: auto; padding: 15px; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 10px; text-decoration: none;">
                        <span class="dashicons dashicons-admin-settings" style="font-size: 32px;"></span>
                        <?php _e('Configure Settings', 'product-bundle-quantity'); ?>
                    </a>
                    <a href="<?php echo admin_url('edit.php?post_type=product&meta_key=_enable_bundle_quantity&meta_value=yes'); ?>" class="button button-secondary" style="height: auto; padding: 15px; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 10px; text-decoration: none;">
                        <span class="dashicons dashicons-list-view" style="font-size: 32px;"></span>
                        <?php _e('View Bundle Products', 'product-bundle-quantity'); ?>
                    </a>
                    <?php if (!$is_premium): ?>
                    <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>" class="button button-primary" style="height: auto; padding: 15px; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 10px; text-decoration: none; background: #ffc107; border-color: #ffc107; color: #000;">
                        <span class="dashicons dashicons-unlock" style="font-size: 32px;"></span>
                        <?php _e('Activate Premium', 'product-bundle-quantity'); ?>
                    </a>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Features Overview -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin: 20px 0;">
                <!-- Free Features -->
                <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                    <h2 style="margin: 0 0 15px 0; display: flex; align-items: center; gap: 10px;">
                        <span class="dashicons dashicons-yes" style="color: #46b450;"></span>
                        <?php _e('Free Features', 'product-bundle-quantity'); ?>
                    </h2>
                    <ul style="margin: 0; padding-left: 20px;">
                        <li><?php _e('Bundle quantity options for products', 'product-bundle-quantity'); ?></li>
                        <li><?php _e('Custom images for bundles', 'product-bundle-quantity'); ?></li>
                        <li><?php _e('Global color settings', 'product-bundle-quantity'); ?></li>
                        <li><?php _e('Display options control', 'product-bundle-quantity'); ?></li>
                        <li><?php _e('Theme color auto-detection', 'product-bundle-quantity'); ?></li>
                        <li><?php _e('WooCommerce HPOS compatibility', 'product-bundle-quantity'); ?></li>
                    </ul>
                </div>

                <!-- Premium Features -->
                <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                    <h2 style="margin: 0 0 15px 0; display: flex; align-items: center; gap: 10px;">
                        <span class="dashicons dashicons-star-filled" style="color: #ffc107;"></span>
                        <?php _e('Premium Features', 'product-bundle-quantity'); ?>
                    </h2>
                    <ul style="margin: 0; padding-left: 20px;">
                        <li>
                            <strong><?php _e('Custom Colors', 'product-bundle-quantity'); ?></strong>
                            <?php if (!$is_premium) echo '<span class="dashicons dashicons-lock" style="color: #d63638; font-size: 14px;"></span>'; ?>
                            <br><small style="color: #646970;"><?php _e('Per-product color customization', 'product-bundle-quantity'); ?></small>
                        </li>
                        <li>
                            <strong><?php _e('Product Discounts', 'product-bundle-quantity'); ?></strong>
                            <?php if (!$is_premium) echo '<span class="dashicons dashicons-lock" style="color: #d63638; font-size: 14px;"></span>'; ?>
                            <br><small style="color: #646970;"><?php _e('Set discount percentages on bundles', 'product-bundle-quantity'); ?></small>
                        </li>
                        <li>
                            <strong><?php _e('Priority Support', 'product-bundle-quantity'); ?></strong>
                            <?php if (!$is_premium) echo '<span class="dashicons dashicons-lock" style="color: #d63638; font-size: 14px;"></span>'; ?>
                            <br><small style="color: #646970;"><?php _e('Get help from our expert team', 'product-bundle-quantity'); ?></small>
                        </li>
                    </ul>
                    <?php if (!$is_premium): ?>
                        <a href="<?php echo admin_url('admin.php?page=kbq-license'); ?>" class="button button-primary" style="width: 100%; margin-top: 15px; text-align: center;">
                            <?php _e('Unlock Premium Features', 'product-bundle-quantity'); ?>
                        </a>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Getting Started -->
            <div style="background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin: 20px 0; box-shadow: 0 1px 1px rgba(0,0,0,0.04);">
                <h2 style="margin: 0 0 15px 0;"><?php _e('Getting Started', 'product-bundle-quantity'); ?></h2>
                <ol style="margin: 0; padding-left: 20px; line-height: 1.8;">
                    <li><?php _e('Go to any product edit page in WooCommerce', 'product-bundle-quantity'); ?></li>
                    <li><?php _e('Navigate to the "Bundle Quantities" tab', 'product-bundle-quantity'); ?></li>
                    <li><?php _e('Check "Enable Bundle Quantity" to activate bundles for that product', 'product-bundle-quantity'); ?></li>
                    <li><?php _e('Add bundle options with quantity, discount, and images', 'product-bundle-quantity'); ?></li>
                    <li><?php _e('Customize colors globally or per-product (Premium)', 'product-bundle-quantity'); ?></li>
                    <li><?php _e('Save and view your product page to see the bundle options', 'product-bundle-quantity'); ?></li>
                </ol>
            </div>

            <!-- Support & Resources -->
            <div style="background: #f0f6fc; border: 1px solid #c3d9f0; border-radius: 4px; padding: 20px; margin: 20px 0;">
                <h2 style="margin: 0 0 15px 0;"><?php _e('Support & Resources', 'product-bundle-quantity'); ?></h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                    <a href="https://codemoksha.com/docs" target="_blank" class="button button-secondary" style="text-decoration: none; display: flex; align-items: center; gap: 10px; justify-content: center;">
                        <span class="dashicons dashicons-book"></span>
                        <?php _e('Documentation', 'product-bundle-quantity'); ?>
                    </a>
                    <a href="https://codemoksha.com/support" target="_blank" class="button button-secondary" style="text-decoration: none; display: flex; align-items: center; gap: 10px; justify-content: center;">
                        <span class="dashicons dashicons-sos"></span>
                        <?php _e('Get Support', 'product-bundle-quantity'); ?>
                    </a>
                    <a href="https://codemoksha.com" target="_blank" class="button button-secondary" style="text-decoration: none; display: flex; align-items: center; gap: 10px; justify-content: center;">
                        <span class="dashicons dashicons-admin-site"></span>
                        <?php _e('Visit Website', 'product-bundle-quantity'); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'product-bundle-quantity'));
        }

        $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'colors';
        $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());

        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <h2 class="nav-tab-wrapper">
                <a href="?page=product-bundle-quantity-settings&tab=colors"
                   class="nav-tab <?php echo $active_tab === 'colors' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Colors', 'product-bundle-quantity'); ?>
                </a>
                <a href="?page=product-bundle-quantity-settings&tab=display"
                   class="nav-tab <?php echo $active_tab === 'display' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Display Options', 'product-bundle-quantity'); ?>
                </a>
            </h2>

            <?php settings_errors('product_bundle_quantity_settings'); ?>

            <form method="post" action="options.php">
                <?php
                settings_fields('product_bundle_quantity_settings_group');

                if ($active_tab === 'colors') {
                    $this->render_colors_tab($settings);
                } else {
                    $this->render_display_tab($settings);
                }

                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render the colors tab
     *
     * @param array $settings Current settings
     */
    private function render_colors_tab($settings) {
        $colors = $settings['colors'];
        $theme_detected = isset($settings['theme_colors_detected']) ? $settings['theme_colors_detected'] : false;
        ?>
        <div class="kbq-settings-tab">
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Theme Integration', 'product-bundle-quantity'); ?></th>
                    <td>
                        <button type="button" id="kbq-detect-theme-colors" class="button button-secondary">
                            <?php _e('Auto-Detect Theme Colors', 'product-bundle-quantity'); ?>
                        </button>
                        <a href="<?php echo wp_nonce_url(add_query_arg(array('kbq_action' => 'reset_colors', 'tab' => 'colors')), 'kbq_reset_colors_action'); ?>"
                           id="kbq-reset-colors"
                           class="button button-secondary"
                           style="margin-left: 10px;"
                           onclick="return confirm('<?php echo esc_js(__('Are you sure you want to reset all colors to their default values? This action cannot be undone.', 'product-bundle-quantity')); ?>');">
                            <?php _e('Reset to Default Colors', 'product-bundle-quantity'); ?>
                        </a>
                        <?php if (isset($_GET['kbq_reset']) && $_GET['kbq_reset'] === 'success'): ?>
                            <span class="description" style="color: green; margin-left: 10px; font-weight: bold;">
                                ✓ <?php _e('Colors have been successfully reset to defaults!', 'product-bundle-quantity'); ?>
                            </span>
                        <?php endif; ?>
                        <?php if ($theme_detected): ?>
                            <span class="description" style="color: green; margin-left: 10px;">
                                ✓ <?php _e('Theme colors detected and applied', 'product-bundle-quantity'); ?>
                            </span>
                        <?php endif; ?>
                        <p class="description">
                            <?php _e('Automatically detect and apply your active theme\'s color palette, or reset all colors to their original default values.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <h3><?php _e('Bundle Selection Colors', 'product-bundle-quantity'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="kbq_color_selected_border">
                            <?php _e('Selected Bundle Border', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_selected_border"
                               name="product_bundle_quantity_settings[colors][selected_border]"
                               value="<?php echo esc_attr($colors['selected_border']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#f7dc5a" />
                        <p class="description">
                            <?php _e('Border color when a bundle is selected.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_selected_bg">
                            <?php _e('Selected Bundle Background', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_selected_bg"
                               name="product_bundle_quantity_settings[colors][selected_bg]"
                               value="<?php echo esc_attr($colors['selected_bg']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#fff1a4" />
                        <p class="description">
                            <?php _e('Background color when a bundle is selected.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_default_border">
                            <?php _e('Default Bundle Border', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_default_border"
                               name="product_bundle_quantity_settings[colors][default_border]"
                               value="<?php echo esc_attr($colors['default_border']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#e0e0e0" />
                        <p class="description">
                            <?php _e('Border color for unselected bundles.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <h3><?php _e('Button & Badge Colors', 'product-bundle-quantity'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="kbq_color_button_bg">
                            <?php _e('Add to Cart Button', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_button_bg"
                               name="product_bundle_quantity_settings[colors][button_bg]"
                               value="<?php echo esc_attr($colors['button_bg']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#ffc00b" />
                        <p class="description">
                            <?php _e('Background color for the Add to Cart button.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_button_text">
                            <?php _e('Button Text Color', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_button_text"
                               name="product_bundle_quantity_settings[colors][button_text]"
                               value="<?php echo esc_attr($colors['button_text']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#ffffff" />
                        <p class="description">
                            <?php _e('Text color for the Add to Cart button.', 'product-bundle-quantity'); ?>
                        </p>
                        <div id="kbq_button_contrast_warning" class="kbq-contrast-warning" style="display:none;">
                            <span class="dashicons dashicons-warning" style="color: #f0b849;"></span>
                            <span class="message"></span>
                        </div>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_saving_badge">
                            <?php _e('Saving Badge', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_saving_badge"
                               name="product_bundle_quantity_settings[colors][saving_badge]"
                               value="<?php echo esc_attr($colors['saving_badge']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#ff0000" />
                        <p class="description">
                            <?php _e('Background color for the savings percentage badge.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <h3><?php _e('Text Colors', 'product-bundle-quantity'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="kbq_color_price_text">
                            <?php _e('Price Text', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_price_text"
                               name="product_bundle_quantity_settings[colors][price_text]"
                               value="<?php echo esc_attr($colors['price_text']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#333333" />
                        <p class="description">
                            <?php _e('Color for price text.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_secondary_text">
                            <?php _e('Secondary Text', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_secondary_text"
                               name="product_bundle_quantity_settings[colors][secondary_text]"
                               value="<?php echo esc_attr($colors['secondary_text']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#666666" />
                        <p class="description">
                            <?php _e('Color for secondary text (stock info, security message, "Per Unit").', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="kbq_color_muted_text">
                            <?php _e('Muted Text', 'product-bundle-quantity'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text"
                               id="kbq_color_muted_text"
                               name="product_bundle_quantity_settings[colors][muted_text]"
                               value="<?php echo esc_attr($colors['muted_text']); ?>"
                               class="kbq-color-picker"
                               data-default-color="#999999" />
                        <p class="description">
                            <?php _e('Color for strikethrough original price.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }

    /**
     * Render the display options tab
     *
     * @param array $settings Current settings
     */
    private function render_display_tab($settings) {
        $display = $settings['display'];
        ?>
        <div class="kbq-settings-tab">
            <!-- Hidden field to detect we're saving from display tab -->
            <input type="hidden" name="product_bundle_quantity_settings[display][_tab_marker]" value="1" />

            <h3><?php _e('Frontend Display Options', 'product-bundle-quantity'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <?php _e('Stock Label', 'product-bundle-quantity'); ?>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox"
                                   id="kbq_show_stock_label"
                                   name="product_bundle_quantity_settings[display][show_stock_label]"
                                   value="1"
                                   <?php checked($display['show_stock_label'], true); ?> />
                            <?php _e('Show "In Stock" label', 'product-bundle-quantity'); ?>
                        </label>
                        <p class="description">
                            <?php _e('Display the "✓ In Stock" label below the Add to Cart button when product is available.', 'product-bundle-quantity'); ?>
                        </p>
                        <div class="kbq-preview-box" style="margin-top: 10px; padding: 10px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px;">
                            <strong><?php _e('Preview:', 'product-bundle-quantity'); ?></strong><br>
                            <span style="color: #666;">✓ <?php _e('In Stock', 'product-bundle-quantity'); ?></span>
                        </div>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <?php _e('Security Message', 'product-bundle-quantity'); ?>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox"
                                   id="kbq_show_security_message"
                                   name="product_bundle_quantity_settings[display][show_security_message]"
                                   value="1"
                                   <?php checked($display['show_security_message'], true); ?> />
                            <?php _e('Show security message', 'product-bundle-quantity'); ?>
                        </label>
                        <p class="description">
                            <?php _e('Display the "🔒 All transactions secured and encrypted" message below the stock label.', 'product-bundle-quantity'); ?>
                        </p>
                        <div class="kbq-preview-box" style="margin-top: 10px; padding: 10px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px;">
                            <strong><?php _e('Preview:', 'product-bundle-quantity'); ?></strong><br>
                            <span style="color: #666;">🔒 <?php _e('All transactions secured and encrypted', 'product-bundle-quantity'); ?></span>
                        </div>
                    </td>
                </tr>
            </table>

            <h3><?php _e('Per-Product Customization', 'product-bundle-quantity'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <?php _e('Product-Level Overrides', 'product-bundle-quantity'); ?>
                    </th>
                    <td>
                        <p class="description">
                            <?php _e('These global settings apply to all products with bundle quantities enabled. You can override colors for individual products in the product edit screen under the "Bundle Quantities" tab.', 'product-bundle-quantity'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }




    public function remove_default_woo_actions() {
        if (!is_product()) {
            return; // Only run on single product pages
        }

        $product = wc_get_product();

        if (!$product) {
            return; // Exit if no product is found
        }

        $enable_bundle_quantity = get_post_meta($product->get_id(), '_enable_bundle_quantity', true);
        if ($enable_bundle_quantity !== 'yes') return;

        // Remove the default Add to Cart button and quantity input
        remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30);
        remove_action('woocommerce_simple_add_to_cart', 'woocommerce_simple_add_to_cart', 30);
        remove_action('woocommerce_grouped_add_to_cart', 'woocommerce_grouped_add_to_cart', 30);
        remove_action('woocommerce_variable_add_to_cart', 'woocommerce_variable_add_to_cart', 30);
        remove_action('woocommerce_external_add_to_cart', 'woocommerce_external_add_to_cart', 30);

        add_action('woocommerce_single_product_summary', array($this, 'display_bundle_options'), 35);
    }



    public function display_bundle_options($product_id = 0) {
        global $product;

        // Try to get product from multiple sources with proper fallbacks
        if ($product_id > 0) {
            // Use provided product ID
            $current_product = wc_get_product($product_id);
        } elseif ($product && is_a($product, 'WC_Product')) {
            // Use global product if available
            $current_product = $product;
        } elseif (is_product()) {
            // Try to get product from current post on product pages
            $current_product = wc_get_product(get_the_ID());
        } else {
            // No valid product found
            return;
        }

        // Final validation - ensure we have a valid product object
        if (!$current_product || !is_a($current_product, 'WC_Product')) {
            return;
        }

        $product_id = $current_product->get_id();
        $enable_bundle_quantity = get_post_meta($product_id, '_enable_bundle_quantity', true);
        if ($enable_bundle_quantity !== 'yes') return;

        $bundle_options = get_post_meta($product_id, '_bundle_options', true);
        if (!$bundle_options) return;

        // Check if user has premium access for product discounts
        $has_discount_access = $this->has_premium_access('product_discounts');

        // Detect if we're in a shortcode context (not on a product page)
        // In shortcode context, we need to inject product-specific CSS variables
        $is_shortcode_context = !is_product();
        $unique_id = '';
        $wrapper_class = '';

        if ($is_shortcode_context) {
            // Generate unique ID for this bundle instance
            $unique_id = 'kbq-bundle-' . $product_id . '-' . uniqid();
            $wrapper_class = 'kbq-shortcode-wrapper-' . $product_id;

            // Output inline CSS with product-specific variables scoped to this instance
            // Also add wrapper styles to match WooCommerce product page layout
            echo '<style type="text/css">';
            echo $this->generate_product_specific_css_variables($product_id, '#' . $unique_id);

            // Add comprehensive layout styles to match WooCommerce product pages
            echo '.kbq-shortcode-wrapper {';
            echo 'width: 100%;';
            echo 'max-width: 100%;';
            echo 'margin: 0 auto 2em;';
            echo 'padding: 0;';
            echo 'box-sizing: border-box;';
            echo 'clear: both;';
            echo '}';

            echo '.kbq-shortcode-wrapper .summary.entry-summary {';
            echo 'width: 100%;';
            echo 'max-width: 100%;';
            echo 'margin: 0;';
            echo 'padding: 0;';
            echo 'float: none;';
            echo 'box-sizing: border-box;';
            echo '}';

            echo '.kbq-shortcode-wrapper .choose-your-package {';
            echo 'width: 100%;';
            echo 'max-width: 100%;';
            echo 'margin: 0 auto;';
            echo 'box-sizing: border-box;';
            echo '}';

            // Ensure proper responsive layout for bundle options
            echo '.kbq-shortcode-wrapper .bundle-options {';
            echo 'display: flex !important;';
            echo 'flex-wrap: wrap;';
            echo 'gap: 15px;';
            echo 'justify-content: space-between;';
            echo '}';

            echo '.kbq-shortcode-wrapper .bundle-option {';
            echo 'flex: 1 1 auto;';
            echo 'min-width: 0;';
            echo 'box-sizing: border-box;';
            echo '}';

            // Ensure button styling matches
            echo '.kbq-shortcode-wrapper .add-to-cart-bundle {';
            echo 'width: 100%;';
            echo 'box-sizing: border-box;';
            echo 'display: block;';
            echo '}';

            // Responsive adjustments for mobile
            echo '@media (max-width: 768px) {';
            echo '.kbq-shortcode-wrapper .bundle-options {';
            echo 'flex-direction: column;';
            echo '}';
            echo '.kbq-shortcode-wrapper .bundle-option {';
            echo 'flex: 1 1 100%;';
            echo '}';
            echo '}';

            echo '</style>';

            // Add WooCommerce-style wrapper to match product page structure
            echo '<div class="kbq-shortcode-wrapper product type-product ' . esc_attr($wrapper_class) . '">';
            echo '<div class="summary entry-summary">';
        }

        ?>
        <div class="choose-your-package" <?php if ($is_shortcode_context) echo 'id="' . esc_attr($unique_id) . '"'; ?>>
            <h2><?php _e('Choose Your Package', 'product-bundle-quantity'); ?></h2>
            <div class="bundle-options">
                <?php foreach ($bundle_options as $index => $option) :
                    // Only apply discount if premium access is available
                    $discount = 0.0;
                    if ($has_discount_access && !empty($option['discount']) && is_numeric($option['discount'])) {
                        $discount = (float) $option['discount'];
                    }
                ?>
                    <div class="bundle-option" data-image="<?php echo $option['image']?wp_get_attachment_url($option['image']):''; ?>" data-index="<?php echo $index; ?>" data-price="<?php echo $current_product->get_price(); ?>" data-quantity="<?php echo esc_attr($option['quantity']); ?>" data-discount="<?php echo esc_attr($discount); ?>">
<!--                       make these 3 under transparent box-->

                        <div class="bundle-image" style="width: 50%;">
                            <img src="<?php echo $option['image']?wp_get_attachment_url($option['image']):'' ?>">
                        </div>
                        <div  class="bundle-option-product-info" >
                        <div class="quantity"><?php echo esc_html($option['quantity']); ?></div>
                        <div class="price"><?php
                            echo wc_price($current_product->get_price() * (1 - $discount / 100));
                        ?></div>
                        <div class="per-unit"><?php _e('Per Unit', 'product-bundle-quantity'); ?></div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            <div class="total-price">
                <?php
                // Get the first bundle option for initial display
                $first_option = $bundle_options[0];

                // Only apply discount if premium access is available
                $discount_value = 0.0;
                if ($has_discount_access && !empty($first_option['discount']) && is_numeric($first_option['discount'])) {
                    $discount_value = (float) $first_option['discount'];
                }

                // Check if discount is meaningful (>= 0.01%)
                $has_discount = $discount_value >= 0.01;

                $base_total = $current_product->get_price() * $first_option['quantity'];
                $discounted_total = $base_total * (1 - ($discount_value / 100));
                ?>

                <?php if ($has_discount): ?>
                    <!-- Original price shown only when there's an actual discount -->
                    <span class="original-price"><?php echo wc_price($base_total); ?></span>
                <?php endif; ?>

                <!-- Always show the final price -->
                <span class="discounted-price"><?php echo wc_price($discounted_total); ?></span>

                <?php if ($has_discount): ?>
                    <!-- Saving badge shown only when discount >= 0.01% -->
                    <span class="saving"><?php printf(__('Saving %s%%', 'product-bundle-quantity'), number_format($discount_value, 2)); ?></span>
                <?php endif; ?>
            </div>
            <button class="add-to-cart-bundle" data-productid="<?php echo $current_product->get_id(); ?>" ><?php _e('ADD TO CART', 'product-bundle-quantity'); ?></button>
            <?php
            $settings = get_option('product_bundle_quantity_settings', $this->get_default_settings());
            $show_stock = isset($settings['display']['show_stock_label']) ? $settings['display']['show_stock_label'] : true;
            $show_security = isset($settings['display']['show_security_message']) ? $settings['display']['show_security_message'] : true;

            if ($show_stock): ?>
                <div class="stock-info">
                    <?php if ($current_product->is_in_stock()) : ?>
                        <span class="in-stock">✓ <?php _e('In Stock', 'product-bundle-quantity'); ?></span>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <?php if ($show_security): ?>
                <div class="secure-transaction">
                    <span>🔒 <?php _e('All transactions secured and encrypted', 'product-bundle-quantity'); ?></span>
                </div>
            <?php endif; ?>
        </div>
        <?php

        // Close WooCommerce-style wrapper divs for shortcode context
        if ($is_shortcode_context) {
            echo '</div>'; // Close .summary.entry-summary
            echo '</div>'; // Close .kbq-shortcode-wrapper
        }
    }


    public function add_cart_item_data($cart_item_data, $product_id, $variation_id) {
        if (isset($_POST['bundle_option'])) {
            $bundle_options = get_post_meta($product_id, '_bundle_options', true);
            $selected_option = $bundle_options[$_POST['bundle_option']];

            // Only apply discount if user has premium access
            $has_discount_access = $this->has_premium_access('product_discounts');
            $discount = 0;
            if ($has_discount_access && !empty($selected_option['discount'])) {
                $discount = $selected_option['discount'];
            }

            $cart_item_data['bundle_option'] = array(
                'quantity' => $selected_option['quantity'],
                'discount' => $discount
            );
        }
        return $cart_item_data;
    }

    public function get_cart_item_from_session($cart_item, $values) {
        if (isset($values['bundle_option'])) {
            $cart_item['bundle_option'] = $values['bundle_option'];
        }
        return $cart_item;
    }

    public function update_cart_item_price($price, $cart_item, $cart_item_key) {
        if (isset($cart_item['bundle_option'])) {
            $product = $cart_item['data'];
            $base_price = $product->get_price();
            $quantity = $cart_item['bundle_option']['quantity'];

            // Validate discount and check premium access
            $discount = 0;
            if (!empty($cart_item['bundle_option']['discount']) && is_numeric($cart_item['bundle_option']['discount'])) {
                $has_discount_access = $this->has_premium_access('product_discounts');
                if ($has_discount_access) {
                    $discount = (float) $cart_item['bundle_option']['discount'];
                }
            }

            $discounted_price = $base_price * (1 - $discount / 100);
            return wc_price($discounted_price);
        }
        return $price;
    }


    public function add_bundle_option_template() {
        $has_discount_access = $this->has_premium_access('product_discounts');
        ?>
        <script type="text/template" id="tmpl-bundle-option-template">
            <div class="bundle-option">
                <p class="form-field">
                    <label><?php _e('Quantity', 'product-bundle-quantity'); ?></label>
                    <input type="number" name="bundle_options[{{data.index}}][quantity]" value="" min="1" step="1" />
                </p>
                <p class="form-field">
                    <label><?php _e('Discount (%)', 'product-bundle-quantity'); ?><?php if (!$has_discount_access) echo ' <span style="color: #856404;">(' . __('Premium', 'product-bundle-quantity') . ')</span>'; ?></label>
                    <input type="number"
                           name="bundle_options[{{data.index}}][discount]"
                           value="0"
                           min="0"
                           max="100"
                           step="0.01"
                           <?php if (!$has_discount_access) echo 'disabled="disabled" readonly'; ?>
                           class="kbq-discount-field" />
                    <?php if (!$has_discount_access): ?>
                        <input type="hidden" name="bundle_options[{{data.index}}][discount]" value="0" />
                    <?php endif; ?>
                </p>

                <p class="form-field">
                    <label><?php _e('Image', 'product-bundle-quantity'); ?></label>
                    <input type="hidden" class="bundle_image_id" name="bundle_options[{{data.index}}][image]" value="" />
                    <img src="" alt="" class="bundle-image-preview" style="max-width: 100px; max-height: 100px;" />
                    <button type="button" class="button upload_image_button" data-index="{{data.index}}"><?php _e('Upload/Add Image', 'product-bundle-quantity'); ?></button>
                    <button type="button" class="button remove_image_button" data-index="{{data.index}}" style="display: none;" ><?php _e('Remove Image', 'product-bundle-quantity'); ?></button>
                </p>

                <button type="button" class="button remove-bundle-option"><?php _e('Remove', 'product-bundle-quantity'); ?></button>
            </div>
        </script>
        <?php
    }

    public function add_bundle_to_cart() {
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $bundle_index = isset($_POST['bundle_index']) ? intval($_POST['bundle_index']) : -1;

        if (!$product_id || $bundle_index < 0) {
            wp_send_json_error('Invalid product or bundle selection.');
            wp_die();
        }

        $bundle_options = get_post_meta($product_id, '_bundle_options', true);

        if (!is_array($bundle_options) || !isset($bundle_options[$bundle_index])) {
            wp_send_json_error('Bundle option not found.');
            wp_die();
        }

        $selected_bundle = $bundle_options[$bundle_index];
        $quantity = intval($selected_bundle['quantity']);

        // Only apply discount if user has premium access
        $has_discount_access = $this->has_premium_access('product_discounts');
        $discount = 0;
        if ($has_discount_access && !empty($selected_bundle['discount']) && is_numeric($selected_bundle['discount'])) {
            $discount = floatval($selected_bundle['discount']);
        }

        $cart_item_data = array(
            'bundle_option' => array(
                'quantity' => $quantity,
                'discount' => $discount,
                'bundle_index' => $bundle_index
            )
        );

        // Add to cart with the bundle quantity
        $added = WC()->cart->add_to_cart($product_id, $quantity, 0, array(), $cart_item_data);

        if ($added) {
            if ($discount > 0) {
                $message = sprintf(__('%d items added to cart with %s%% discount.', 'product-bundle-quantity'), $quantity, $discount);
            } else {
                $message = sprintf(__('%d items added to cart.', 'product-bundle-quantity'), $quantity);
            }
            wp_send_json_success(array('message' => $message));
        } else {
            wp_send_json_error(__('Error adding to cart. Please try again.', 'product-bundle-quantity'));
        }

        wp_die();
    }


    public function calculate_cart_item_price($cart_object) {
        if (is_admin() && !defined('DOING_AJAX')) return;

        foreach ($cart_object->get_cart() as $cart_item_key => $cart_item) {
            if (isset($cart_item['bundle_option'])) {
                $product = $cart_item['data'];
                $base_price = $product->get_price();
                $quantity = $cart_item['bundle_option']['quantity'];

                // Validate discount and check premium access
                $discount = 0;
                if (!empty($cart_item['bundle_option']['discount']) && is_numeric($cart_item['bundle_option']['discount'])) {
                    $has_discount_access = $this->has_premium_access('product_discounts');
                    if ($has_discount_access) {
                        $discount = (float) $cart_item['bundle_option']['discount'];
                    }
                }

                // Calculate the discounted price
                $discounted_price = $base_price * (1 - $discount / 100);

                // Set the new price
                $cart_item['data']->set_price($discounted_price);
            }
        }
    }




    public function update_cart_item_quantity($product_quantity, $cart_item_key, $cart_item) {
        if (isset($cart_item['bundle_option'])) {
            return $cart_item['bundle_option']['quantity'];
        }
        return $product_quantity;
    }

    /**
     * Migrate old data from kund_* to product_* for backward compatibility
     * This runs once and migrates all existing data to the new naming convention
     */
    public function migrate_old_data() {
        // Check if migration has already been done
        if (get_option('pbq_migration_completed')) {
            return;
        }

        // Migrate global settings
        $old_settings = get_option('kund_bundle_quantity_settings');
        if ($old_settings && !get_option('product_bundle_quantity_settings')) {
            update_option('product_bundle_quantity_settings', $old_settings);
        }

        // Migrate product meta data
        global $wpdb;

        // Migrate _kund_bundle_custom_colors_enabled to _product_bundle_custom_colors_enabled
        $wpdb->query("
            UPDATE {$wpdb->postmeta}
            SET meta_key = '_product_bundle_custom_colors_enabled'
            WHERE meta_key = '_kund_bundle_custom_colors_enabled'
        ");

        // Migrate _kund_bundle_custom_colors to _product_bundle_custom_colors
        $wpdb->query("
            UPDATE {$wpdb->postmeta}
            SET meta_key = '_product_bundle_custom_colors'
            WHERE meta_key = '_kund_bundle_custom_colors'
        ");

        // Mark migration as completed
        update_option('pbq_migration_completed', true);
    }


}

new Product_Bundle_Quantity();